<?php

/**
 * Tests of OAuth implementation.
 * 
 * @version $Id$
 * @author Marc Worrell <marc@mediamatic.nl>
 * @copyright (c) 2007 Mediamatic Lab
 * @date  Nov 29, 2007 3:46:56 PM
 * @see http://wiki.oauth.net/TestCases
 */

require_once dirname(__FILE__) . '/../core/OAuthRequest.php';
require_once dirname(__FILE__) . '/../core/OAuthRequester.php';
require_once dirname(__FILE__) . '/../core/OAuthRequestSigner.php';
require_once dirname(__FILE__) . '/../core/OAuthRequestVerifier.php';

function oauth_test ()
{
	header('Content-Type: text/plain; charset=utf-8');
	
	echo "Performing OAuth module tests.\n\n";
	echo "See also: http://wiki.oauth.net/TestCases\n\n";
	
	assert_options(ASSERT_CALLBACK, 'oauth_assert_handler');
	assert_options(ASSERT_WARNING,  0);
	
	$req = new OAuthRequest();

	echo "\n\n***** Parameter Encoding *****\n\n";
	
	assert('$req->urlencode(\'abcABC123\') == \'abcABC123\'');
	assert('$req->urlencode(\'-._~\') == \'-._~\'');
	assert('$req->urlencode(\'%\') == \'%25\'');
	assert('$req->urlencode(\'&=*\') == \'%26%3D%2A\'');
	assert('$req->urlencode(\'&=*\') == \'%26%3D%2A\'');
	assert('$req->urlencode("\n") == \'%0A\'');
	assert('$req->urlencode(" ") == \'%20\'');
	assert('$req->urlencode("\x7f") == \'%7F\'');


	echo "***** Normalize Request Parameters *****\n\n";
	
	$req = new OAuthRequest('http://example.com/?name');
	assert('$req->getNormalizedParams() == \'name=\'');

	$req = new OAuthRequest('http://example.com/?a=b');
	assert('$req->getNormalizedParams() == \'a=b\'');
	
	$req = new OAuthRequest('http://example.com/?a=b&c=d');
	assert('$req->getNormalizedParams() == \'a=b&c=d\'');
	
	// At this moment we don't support two parameters with the same name
	// so I changed this test case to "a=" and "b=" and not "a=" and "a="
	$req = new OAuthRequest('http://example.com/?b=x!y&a=x+y');
	assert('$req->getNormalizedParams() == \'a=x%20y&b=x%21y\'');
	
	$req = new OAuthRequest('http://example.com/?x!y=a&x=a');
	assert('$req->getNormalizedParams() == \'x=a&x%21y=a\'');
	

	echo "***** Base String *****\n\n";
	
	$req = new OAuthRequest('http://example.com?n=v', 'GET');
	assert('$req->signatureBaseString() == \'GET&http%3A%2F%2Fexample.com&n%3Dv\'');
	
	$req = new OAuthRequest(
							'https://photos.example.net/request_token', 
							'POST',
							'oauth_version=1.0&oauth_consumer_key=dpf43f3p2l4k3l03&oauth_timestamp=1191242090&oauth_nonce=hsu94j3884jdopsl&oauth_signature_method=PLAINTEXT&oauth_signature=ignored',
							array('X-OAuth-Test' => true));
	assert('$req->signatureBaseString() == \'POST&https%3A%2F%2Fphotos.example.net%2Frequest_token&oauth_consumer_key%3Ddpf43f3p2l4k3l03%26oauth_nonce%3Dhsu94j3884jdopsl%26oauth_signature_method%3DPLAINTEXT%26oauth_timestamp%3D1191242090%26oauth_version%3D1.0\'');

	$req = new OAuthRequest(
							'http://photos.example.net/photos?file=vacation.jpg&size=original&oauth_version=1.0&oauth_consumer_key=dpf43f3p2l4k3l03&oauth_token=nnch734d00sl2jdk&oauth_timestamp=1191242096&oauth_nonce=kllo9940pd9333jh&oauth_signature=ignored&oauth_signature_method=HMAC-SHA1', 
							'GET');
	assert('$req->signatureBaseString() == \'GET&http%3A%2F%2Fphotos.example.net%2Fphotos&file%3Dvacation.jpg%26oauth_consumer_key%3Ddpf43f3p2l4k3l03%26oauth_nonce%3Dkllo9940pd9333jh%26oauth_signature_method%3DHMAC-SHA1%26oauth_timestamp%3D1191242096%26oauth_token%3Dnnch734d00sl2jdk%26oauth_version%3D1.0%26size%3Doriginal\'');


	echo "***** HMAC-SHA1 *****\n\n";

	
	$req = new OAuthRequestSigner('http://photos.example.net/photos?file=vacation.jpg&size=original', 'GET');	
	$req->selectSignatureMethod(array('HMAC-SHA1'));

	assert('$req->urldecode($req->signature_HMAC_SHA1(\'bs\', \'cs\', \'\')) == \'egQqG5AJep5sJ7anhXju1unge2I=\'');
	assert('$req->urldecode($req->signature_HMAC_SHA1(\'bs\', \'cs\', \'ts\')) == \'VZVjXceV7JgPq/dOTnNmEfO0Fv8=\'');
	
	$secrets = array(
				'consumer_key'		=> 'dpf43f3p2l4k3l03',
				'consumer_secret'	=> 'kd94hf93k423kf44',
				'token'				=> 'nnch734d00sl2jdk',
				'token_secret'		=> 'pfkkdhi9sl3r4s00',
				'signature_methods'	=> array('HMAC-SHA1'),
				'nonce'				=> 'kllo9940pd9333jh',
				'timestamp'			=> '1191242096'
				);
	$req->sign(0, $secrets);
	assert('$req->getParam(\'oauth_signature\', true) == \'tR3+Ty81lMeYAr/Fid0kMTYa/WM=\'');
	
	echo "\n\nFinished.";
}


function oauth_assert_handler ( $file, $line, $code )
{
	echo "\nAssertion failed in $file:$line
   $code\n\n";
}

/* vi:set ts=4 sts=4 sw=4 binary noeol: */

?>