/*
  +----------------------------------------------------------------------+
  | PHP Version 5                                                        |
  +----------------------------------------------------------------------+
  | Copyright (c) 1997-2004 The PHP Group                                |
  +----------------------------------------------------------------------+
  | This source file is subject to version 3.0 of the PHP license,       |
  | that is bundled with this package in the file LICENSE, and is        |
  | available through the world-wide-web at the following url:           |
  | http://www.php.net/license/3_0.txt.                                  |
  | If you did not receive a copy of the PHP license and are unable to   |
  | obtain it through the world-wide-web, please send a note to          |
  | license@php.net so we can mail you a copy immediately.               |
  +----------------------------------------------------------------------+
  | Authors: Antony Dovgal <tony2001@phpclub.net>                        |
  |          Mikael Johansson <mikael AT synd DOT info>                  |
  +----------------------------------------------------------------------+
  | Modification: Marc Worrell <http://www.marcworrell.com/>             |
  |               Added support for depcached with dependency support    |
  +----------------------------------------------------------------------+
*/

/* $Id: depcached.c 30334 2007-08-24 12:01:54Z marc $ */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "php.h"
#include "php_ini.h"
#include <stdio.h>
#include <fcntl.h>
#ifdef HAVE_SYS_FILE_H
#include <sys/file.h>
#endif

#if HAVE_DEPCACHED

#include <zlib.h>
#include <time.h>
#include "ext/standard/info.h"
#include "ext/standard/php_string.h"
#include "ext/standard/php_var.h"
#include "ext/standard/php_smart_str.h"
#include "ext/standard/crc32.h"
#include "php_network.h"
#include "php_depcached.h"

/* True global resources - no need for thread safety here */
static int le_depcached_pool, le_pdepcached;
static zend_class_entry *depcached_class_entry_ptr;

ZEND_DECLARE_MODULE_GLOBALS(depcached)

/* {{{ depcached_functions[]
 */
zend_function_entry depcached_functions[] = {
	PHP_FE(depcached_connect,		NULL)
	PHP_FE(depcached_pconnect,		NULL)
	PHP_FE(depcached_add_server,	NULL)
	PHP_FE(depcached_get_version,	NULL)
	PHP_FE(depcached_set,			NULL)
	PHP_FE(depcached_get,			NULL)
	PHP_FE(depcached_queue,			NULL)
	PHP_FE(depcached_dequeue,		NULL)
	PHP_FE(depcached_delete,		NULL)
	PHP_FE(depcached_debug,			NULL)
	PHP_FE(depcached_close,			NULL)
	PHP_FE(depcached_flush,			NULL)
	PHP_FE(depcached_flush_all,		NULL)
	{NULL, NULL, NULL}
};

static zend_function_entry php_depcached_class_functions[] = {
	PHP_FALIAS(connect,			depcached_connect,			NULL)
	PHP_FALIAS(pconnect,		depcached_pconnect,			NULL)
	PHP_FALIAS(addserver,		depcached_add_server,		NULL)
	PHP_FALIAS(getversion,		depcached_get_version,		NULL)
	PHP_FALIAS(set,				depcached_set,				NULL)
	PHP_FALIAS(get,				depcached_get,				NULL)
	PHP_FALIAS(queue,			depcached_queue,			NULL)
	PHP_FALIAS(dequeue,			depcached_dequeue,			NULL)
	PHP_FALIAS(delete,			depcached_delete,			NULL)
	PHP_FALIAS(close,			depcached_close,			NULL)
	PHP_FALIAS(flush,			depcached_flush,			NULL)
	PHP_FALIAS(flush_all,		depcached_flush_all,		NULL)
	{NULL, NULL, NULL}
};

/* }}} */

/* {{{ depcached_module_entry
 */
zend_module_entry depcached_module_entry = {
#if ZEND_MODULE_API_NO >= 20010901
	STANDARD_MODULE_HEADER,
#endif
	"depcached",
	depcached_functions,
	PHP_MINIT(depcached),
	PHP_MSHUTDOWN(depcached),
	PHP_RINIT(depcached),		/* Replace with NULL if there's nothing to do at request start */
	PHP_RSHUTDOWN(depcached),	/* Replace with NULL if there's nothing to do at request end */
	PHP_MINFO(depcached),
#if ZEND_MODULE_API_NO >= 20010901
	"1.0.1", 					/* Replace with version number for your extension */
#endif
	STANDARD_MODULE_PROPERTIES
};
/* }}} */

#ifdef COMPILE_DL_DEPCACHED
ZEND_GET_MODULE(depcached)
#endif

/* {{{ macros */
#define MMC_PREPARE_KEY(key, key_len) \
	php_strtr(key, key_len, "\t\r\n ", "____", 4); \

#if ZEND_DEBUG

#define MMC_DEBUG(info) \
{\
	mmc_debug info; \
}\

#else

#define MMC_DEBUG(info) \
{\
}\

#endif


/* }}} */

/* {{{ internal function protos */
static void 		_mmc_pool_list_dtor		( zend_rsrc_list_entry * TSRMLS_DC );
static void 		_mmc_pserver_list_dtor	( zend_rsrc_list_entry * TSRMLS_DC );
static mmc_pool_t *	mmc_pool_new			( void );
static void 		mmc_pool_add			( mmc_pool_t *, mmc_t *, unsigned int );
static mmc_t *		mmc_server_new			( char *, int, unsigned short, int, int, int TSRMLS_DC );
static void 		mmc_server_free			( mmc_t * TSRMLS_DC );
static void 		mmc_server_disconnect	( mmc_t * TSRMLS_DC );
static void 		mmc_server_deactivate	( mmc_t * TSRMLS_DC );
static int 			mmc_server_failure		( mmc_t * TSRMLS_DC );
static mmc_t *		mmc_server_find			( mmc_pool_t *, char *, int TSRMLS_DC );
static unsigned int mmc_hash				( char *, int );
static int 			mmc_compress			( char **, int *, char *, int TSRMLS_DC );
static int 			mmc_uncompress			( char **, long *, char *, int );
static int 			mmc_get_pool			( zval *, mmc_pool_t ** TSRMLS_DC );
static int 			mmc_open				( mmc_t *, int, char **, int * TSRMLS_DC );
static mmc_t *		mmc_find_persistent		( char *, int, int, int, int TSRMLS_DC );
static int 			mmc_close				( mmc_t * TSRMLS_DC );
static int 			mmc_readline			( mmc_t * TSRMLS_DC );
static char * 		mmc_get_version			( mmc_t * TSRMLS_DC );
static int 			mmc_str_left			( char *, char *, int, int );
static int 			mmc_sendcmd				( mmc_t *, const char *, int TSRMLS_DC );
static int 			mmc_exec_storage_cmd	( mmc_t *, char *, int, char *, int, int, int, int, char *, int, char *, int TSRMLS_DC );
static int 			mmc_parse_response		( char *, char **, int, int *, int * );
static int 			mmc_exec_retrieval_cmd	( char *, mmc_pool_t *, zval *, zval ** TSRMLS_DC );
static int 			mmc_exec_retrieval_cmd_multi ( char *, mmc_pool_t *, zval *, zval ** TSRMLS_DC );
static int			mmc_read_all_values 	( mmc_t *, mmc_key_value_t ** TSRMLS_DC );
static int 			mmc_read_value			( mmc_t *, char **, zval ** TSRMLS_DC );
static int 			mmc_delete				( mmc_t *, char *, int, int TSRMLS_DC );
static int 			mmc_flush				( mmc_t *, char *, int TSRMLS_DC );
static int 			mmc_flush_all			( mmc_t * TSRMLS_DC );
static void 		php_mmc_store 			( INTERNAL_FUNCTION_PARAMETERS, char *, int );
static void 		php_mmc_connect 		( INTERNAL_FUNCTION_PARAMETERS, int );
/* }}} */

/* {{{ php_depcached_init_globals()
*/
static void php_depcached_init_globals(zend_depcached_globals *depcached_globals_p TSRMLS_DC)
{
	DEPCACHED_G(debug_mode)		  = 0;
	DEPCACHED_G(default_port)	  = MMC_DEFAULT_PORT;
	DEPCACHED_G(num_persistent)	  = 0;
	DEPCACHED_G(compression_level) = Z_DEFAULT_COMPRESSION;
}
/* }}} */

/* {{{ PHP_MINIT_FUNCTION
 */
PHP_MINIT_FUNCTION(depcached)
{
	zend_class_entry depcached_class_entry;
	INIT_CLASS_ENTRY(depcached_class_entry, "Depcached", php_depcached_class_functions);
	depcached_class_entry_ptr = zend_register_internal_class(&depcached_class_entry TSRMLS_CC);

	le_depcached_pool = zend_register_list_destructors_ex(_mmc_pool_list_dtor, NULL, "depcached connection", module_number);
	le_pdepcached     = zend_register_list_destructors_ex(NULL, _mmc_pserver_list_dtor, "persistent depcached connection", module_number);

#ifdef ZTS
	ts_allocate_id(&depcached_globals_id, sizeof(zend_depcached_globals), (ts_allocate_ctor) php_depcached_init_globals, NULL);
#else
	php_depcached_init_globals(&depcached_globals TSRMLS_CC);
#endif

	REGISTER_LONG_CONSTANT("depcached_COMPRESSED",MMC_COMPRESSED, CONST_CS | CONST_PERSISTENT);

	return SUCCESS;
}
/* }}} */

/* {{{ PHP_MSHUTDOWN_FUNCTION
 */
PHP_MSHUTDOWN_FUNCTION(depcached)
{
	return SUCCESS;
}
/* }}} */

/* {{{ PHP_RINIT_FUNCTION
 */
PHP_RINIT_FUNCTION(depcached)
{
	DEPCACHED_G(debug_mode) = 0;
	return SUCCESS;
}
/* }}} */

/* {{{ PHP_RSHUTDOWN_FUNCTION
 */
PHP_RSHUTDOWN_FUNCTION(depcached)
{
	return SUCCESS;
}
/* }}} */

/* {{{ PHP_MINFO_FUNCTION
 */
PHP_MINFO_FUNCTION(depcached)
{
	char buf[16];

	sprintf(buf, "%ld", DEPCACHED_G(num_persistent));

	php_info_print_table_start();
	php_info_print_table_header(2, "depcached support", "enabled");
	php_info_print_table_row(2, "Active persistent connections", buf);
	php_info_print_table_row(2, "Revision", "$LastChangedRevision: 30334 $ (Based on memcache revision: 1.39)");
	php_info_print_table_row(2, "Note", "Modified version for depcached with directory support");
	php_info_print_table_end();
}
/* }}} */

/* ------------------
   internal functions
   ------------------ */

static void _mmc_pool_list_dtor(zend_rsrc_list_entry *rsrc TSRMLS_DC) /* {{{ */
{
	mmc_pool_t *pool = (mmc_pool_t *)rsrc->ptr;
	int 		i;
	
	for (i=0; i<pool->num_servers; i++) 
	{
		if (!pool->servers[i]->persistent) 
		{
			mmc_server_free(pool->servers[i] TSRMLS_CC);
		}
	}

	if (pool->num_servers) 
	{
		efree(pool->servers);
		efree(pool->requests);
	}

	if (pool->num_buckets) 
	{
		efree(pool->buckets);
	}

	efree(pool);
}
/* }}} */

static void _mmc_pserver_list_dtor(zend_rsrc_list_entry *rsrc TSRMLS_DC) /* {{{ */
{
	mmc_server_free((mmc_t *)rsrc->ptr TSRMLS_CC);
}
/* }}} */

static mmc_t *mmc_server_new(char *host, int host_len, unsigned short port, int persistent, int timeout, int retry_interval TSRMLS_DC) /* {{{ */
{
	mmc_t *mmc;

	if (persistent) 
	{
		mmc		  = malloc(sizeof(mmc_t));
		mmc->host = malloc(host_len + 1);
	}
	else
	{
		mmc		  = emalloc(sizeof(mmc_t));
		mmc->host = emalloc(host_len + 1);
	}

	memset(&(mmc->outbuf), 0, sizeof(smart_str));
	strncpy(mmc->host, host, host_len);

	mmc->host[host_len] = '\0';
	mmc->stream 		= NULL;
	mmc->status 		= MMC_STATUS_DISCONNECTED;
	mmc->port			= port;
	mmc->persistent		= persistent;
	mmc->timeout		= timeout;
	mmc->retry_interval = retry_interval;

	if (persistent)
	{
		DEPCACHED_G(num_persistent)++;
	}
	return mmc;
}
/* }}} */

static void mmc_server_free(mmc_t *mmc TSRMLS_DC) /* {{{ */
{
	if (mmc->persistent) 
	{
		free(mmc->host);
		free(mmc);
		DEPCACHED_G(num_persistent)--;
	}
	else 
	{
		if (mmc->stream != NULL) 
		{
			php_stream_close(mmc->stream);
		}
		efree(mmc->host);
		efree(mmc);
	}
}
/* }}} */

static mmc_pool_t *mmc_pool_new() /* {{{ */
{
	mmc_pool_t *pool;
	
	pool					   = emalloc(sizeof(mmc_pool_t));
	pool->num_servers 		   = 0;
	pool->num_buckets 		   = 0;
	pool->compress_threshold   = 0;
	pool->min_compress_savings = MMC_DEFAULT_SAVINGS;
	return pool;
}
/* }}} */

static void mmc_pool_add(mmc_pool_t *pool, mmc_t *mmc, unsigned int weight) /* {{{ */
{
	int i;

	/* add server and a preallocated request pointer */
	if (pool->num_servers) 
	{
		pool->servers  = erealloc(pool->servers,  sizeof(mmc_t *) * (pool->num_servers + 1));
		pool->requests = erealloc(pool->requests, sizeof(mmc_t *) * (pool->num_servers + 1));
	}
	else 
	{
		pool->servers  = emalloc(sizeof(mmc_t *));
		pool->requests = emalloc(sizeof(mmc_t *));
	}

	pool->servers[pool->num_servers] = mmc;
	pool->num_servers++;

	/* add weight number of buckets for this server */
	if (pool->num_buckets) 
	{
		pool->buckets = erealloc(pool->buckets, sizeof(mmc_t *) * (pool->num_buckets + weight));
	}
	else 
	{
		pool->buckets = emalloc(sizeof(mmc_t *) * (pool->num_buckets + weight));
	}

	for (i=0; i<weight; i++) 
	{
		pool->buckets[pool->num_buckets + i] = mmc;
	}
	pool->num_buckets += weight;
}
/* }}} */

static int mmc_compress(char **result_data, int *result_len, char *data, int data_len TSRMLS_DC) /* {{{ */
{
	int status;
	int	level = DEPCACHED_G(compression_level);

	*result_len  = data_len + (data_len / 1000) + 15 + 1; /* some magic from zlib.c */
	*result_data = (char *) emalloc(*result_len);

	if (!*result_data) 
	{
		return 0;
	}

	if (level >= 0)
	{
		status = compress2(*result_data, (unsigned long *)result_len, data, data_len, level);
	}
	else
	{
		status = compress(*result_data, (unsigned long *)result_len, data, data_len);
	}

	if (status == Z_OK) 
	{
		*result_data = erealloc(*result_data, *result_len + 1);
		(*result_data)[*result_len] = '\0';
		return 1;
	}
	else
	{
		efree(*result_data);
		return 0;
	}
}
/* }}}*/

static int mmc_uncompress(char **result_data, long *result_len, char *data, int data_len) /* {{{ */
{
	int status;
	unsigned int factor=1, maxfactor=16;
	char *tmp1=NULL;

	do {
		*result_len = (unsigned long)data_len * (1 << factor++);
		*result_data = (char *) erealloc(tmp1, *result_len);
		status = uncompress(*result_data, result_len, data, data_len);
		tmp1 = *result_data;
	} while ((status == Z_BUF_ERROR) && (factor < maxfactor));

	if (status == Z_OK) {
		*result_data = erealloc(*result_data, *result_len + 1);
		(*result_data)[ *result_len ] = '\0';
		return 1;
	} else {
		efree(*result_data);
		return 0;
	}
}
/* }}}*/

static void mmc_debug( const char *format, ...) /* {{{ */
{
	TSRMLS_FETCH();

	if (DEPCACHED_G(debug_mode)) 
	{
		char	buffer[2048];
		va_list args;

		va_start(args, format);
		vsnprintf(buffer, sizeof(buffer)-1, format, args);
		va_end(args);
		buffer[sizeof(buffer)-1] = '\0';
		php_printf("%s<br />\n", buffer);
	}
}
/* }}} */

/**
 * New style crc32 hash, compatible with other clients
 */
static unsigned int mmc_hash(char *key, int key_len) 
{ /* {{{ */
	unsigned int crc = ~0;
	int i;

	for (i=0; i<key_len; i++) 
	{
		CRC32(crc, key[i]);
	}

	crc = (~crc >> 16) & 0x7fff;
  	return crc ? crc : 1;
}
/* }}} */

static int mmc_get_pool(zval *id, mmc_pool_t **pool TSRMLS_DC) /* {{{ */
{
	zval **	connection;
	int 	resource_type;

	if (	Z_TYPE_P(id) != IS_OBJECT 
		||	zend_hash_find(Z_OBJPROP_P(id), "connection", sizeof("connection"), (void **)&connection) == FAILURE) 
	{
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "cannot find connection identifier");
		return 0;
	}

	*pool = (mmc_pool_t *) zend_list_find(Z_LVAL_PP(connection), &resource_type);

	if (	!*pool
		||	resource_type != le_depcached_pool) 
	{
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "connection identifier not found");
		return 0;
	}

	return Z_LVAL_PP(connection);
}
/* }}} */

static int _mmc_open(mmc_t *mmc, char **error_string, int *errnum TSRMLS_DC) /* {{{ */
{
	struct timeval 	tv;
	char *			hostname 	= NULL;
	char *			hash_key 	= NULL;
	char *			errstr   	= NULL;
	int				hostname_len;
	int				err			= 0;

	/* close open stream */
	if (mmc->stream != NULL)
	{
		mmc_server_disconnect(mmc TSRMLS_CC);
	}

	tv.tv_sec    = mmc->timeout;
	tv.tv_usec   = 0;

	hostname     = emalloc(strlen(mmc->host) + MAX_LENGTH_OF_LONG + 1 + 1);
	hostname_len = sprintf(hostname, "%s:%d", mmc->host, mmc->port);

	if (mmc->persistent)
	{
		hash_key = emalloc(sizeof("mmc_open___") - 1 + hostname_len + 1);
		sprintf(hash_key, "mmc_open___%s", hostname);
	}

#if PHP_API_VERSION > 20020918
	mmc->stream = php_stream_xport_create( hostname, hostname_len,
										   ENFORCE_SAFE_MODE | REPORT_ERRORS,
										   STREAM_XPORT_CLIENT | STREAM_XPORT_CONNECT,
										   hash_key, &tv, NULL, &errstr, &err);
#else
	if (mmc->persistent) 
	{
		switch(php_stream_from_persistent_id(hash_key, &(mmc->stream) TSRMLS_CC)) 
		{
			case PHP_STREAM_PERSISTENT_SUCCESS:
				if (php_stream_eof(mmc->stream)) 
				{
					php_stream_pclose(mmc->stream);
					mmc->stream = NULL;
					break;
				}
			case PHP_STREAM_PERSISTENT_FAILURE:
				break;
		}
	}

	if (!mmc->stream) 
	{
		int socktype = SOCK_STREAM;
		mmc->stream = php_stream_sock_open_host(mmc->host, mmc->port, socktype, &tv, hash_key);
	}

#endif

	efree(hostname);
	if (mmc->persistent)
	{
		efree(hash_key);
	}

	if (!mmc->stream) 
	{
		MMC_DEBUG(("_mmc_open: can't open socket to host"));
		mmc_server_deactivate(mmc TSRMLS_CC);

		if (errstr) 
		{
			if (error_string) 
			{
				*error_string = errstr;
			}
			else
			{
				efree(errstr);
			}
		}
		if (errnum)
		{
			*errnum = err;
		}
		return 0;
	}

	php_stream_auto_cleanup(mmc->stream);
	php_stream_set_option(mmc->stream, PHP_STREAM_OPTION_READ_TIMEOUT, 0, &tv);
	php_stream_set_option(mmc->stream, PHP_STREAM_OPTION_WRITE_BUFFER, PHP_STREAM_BUFFER_NONE, NULL);

	/* This should be an INI setting, for now hard-coded to test the results */
	php_stream_set_chunk_size(mmc->stream, 65536);

	mmc->status = MMC_STATUS_CONNECTED;
	return 1;
}
/* }}} */

static int mmc_open(mmc_t *mmc, int force_connect, char **error_string, int *errnum TSRMLS_DC) /* {{{ */
{
	switch (mmc->status) 
	{
	case MMC_STATUS_DISCONNECTED:
		return _mmc_open(mmc, error_string, errnum TSRMLS_CC);

	case MMC_STATUS_CONNECTED:
		return 1;

	case MMC_STATUS_UNKNOWN:
		/* check connection if needed */
		if (force_connect) 
		{
			char *version;
			
			version = mmc_get_version(mmc TSRMLS_CC);
			if (	version == NULL 
				&&	!_mmc_open(mmc, error_string, errnum TSRMLS_CC)) 
			{
				break;
			}
			efree(version);
			mmc->status = MMC_STATUS_CONNECTED;
		}
		return 1;

	case MMC_STATUS_FAILED:
		/* retry failed server, possibly stale cache should be flushed if connect ok
		 * TODO: use client callback on successful reconnect to allow user to specify behaviour
		 */
		if (mmc->retry <= (long)time(NULL)) 
		{
			if (_mmc_open(mmc, error_string, errnum TSRMLS_CC) /*&& mmc_flush(mmc TSRMLS_CC) > 0*/) 
			{
				return 1;
			}
			mmc_server_deactivate(mmc TSRMLS_CC);
		}
		break;
	}
	return 0;
}
/* }}} */

static void mmc_server_disconnect(mmc_t *mmc TSRMLS_DC) /* {{{ */
{
	if (mmc->stream != NULL) 
	{
		if (mmc->persistent) 
		{
			php_stream_pclose(mmc->stream);
		}
		else
		{
			php_stream_close(mmc->stream);
		}
		mmc->stream = NULL;
	}
	mmc->status = MMC_STATUS_DISCONNECTED;
}
/* }}} */

static void mmc_server_deactivate(mmc_t *mmc TSRMLS_DC) /* {{{ */
{
	mmc_server_disconnect(mmc TSRMLS_CC);
	mmc->status = MMC_STATUS_FAILED;
	mmc->retry  = (long) time(NULL) + mmc->retry_interval;
}

/* }}} */

/**
 * Indicate a server failure
 * @return	bool	True if the server was actually deactivated
 */
static int mmc_server_failure(mmc_t *mmc TSRMLS_DC) /* {{{ */
{
	switch (mmc->status) 
	{
	case MMC_STATUS_DISCONNECTED:
		return 0;

	/* attempt reconnect of sockets in unknown state */
	case MMC_STATUS_UNKNOWN:
		mmc->status = MMC_STATUS_DISCONNECTED;
		return 0;
	
	default:
		mmc_server_deactivate(mmc TSRMLS_CC);
		return 1;
	}
}
/* }}} */

static mmc_t *mmc_server_find(mmc_pool_t *pool, char *key, int key_len TSRMLS_DC) /* {{{ */
{
	mmc_t *mmc;

	if (pool->num_servers > 1) 
	{
		unsigned int hash;
		unsigned int i;
		
		hash = mmc_hash(key, key_len);
		mmc  = pool->buckets[hash % pool->num_buckets];

		/* perform failover if needed */
		for (i=0; !mmc_open(mmc, 0, NULL, NULL TSRMLS_CC) && (i<20 || i<pool->num_buckets); i++) 
		{
			char *next_key;
			
			int   next_len;
			
			next_key = emalloc(key_len + MAX_LENGTH_OF_LONG + 1);
			next_len = sprintf(next_key, "%d%s", i+1, key);
		
			MMC_DEBUG(("mmc_server_find: failed to connect to server '%s:%d' status %d, trying next", mmc->host, mmc->port, mmc->status));

			hash += mmc_hash(next_key, next_len);
			mmc   = pool->buckets[hash % pool->num_buckets];

			efree(next_key);
		}
	}
	else 
	{
		mmc = pool->servers[0];
		mmc_open(mmc, 0, NULL, NULL TSRMLS_CC);
	}

	return mmc->status != MMC_STATUS_FAILED ? mmc : NULL;
}
/* }}} */

static int mmc_close(mmc_t *mmc TSRMLS_DC) /* {{{ */
{
	MMC_DEBUG(("mmc_close: closing connection to server"));
	if (!mmc->persistent) 
	{
		mmc_server_disconnect(mmc TSRMLS_CC);
	}
	return 1;
}
/* }}} */

static int mmc_readline(mmc_t *mmc TSRMLS_DC) /* {{{ */
{
	char *buf;

	if (mmc->stream == NULL) 
	{
		MMC_DEBUG(("mmc_readline: socket is already closed"));
		return -1;
	}

	buf = php_stream_gets(mmc->stream, mmc->inbuf, MMC_BUF_SIZE);
	if (buf) 
	{
		MMC_DEBUG(("mmc_readline: read data:"));
		MMC_DEBUG(("mmc_readline:---"));
		MMC_DEBUG(("%s", buf));
		MMC_DEBUG(("mmc_readline:---"));
		return strlen(buf);
	}
	else {
		MMC_DEBUG(("mmc_readline: cannot read a line from the server"));
		return -1;
	}
}
/* }}} */

static char *mmc_get_version(mmc_t *mmc TSRMLS_DC) /* {{{ */
{
	char *version_str;
	int   len;

	if (mmc_sendcmd(mmc, "version", sizeof("version") - 1 TSRMLS_CC) < 0) 
	{
		return NULL;
	}

	len = mmc_readline(mmc TSRMLS_CC);
	if (len < 0) 
	{
		return NULL;
	}

	if (mmc_str_left(mmc->inbuf,"VERSION ", len, sizeof("VERSION ") - 1)) 
	{
		version_str = estrndup(mmc->inbuf + sizeof("VERSION ") - 1, len - (sizeof("VERSION ") - 1) - (sizeof("\r\n") - 1) );
		return version_str;
	}

	MMC_DEBUG(("mmc_get_version: data is not valid version string"));
	return NULL;
}
/* }}} */

static int mmc_str_left(char *haystack, char *needle, int haystack_len, int needle_len) /* {{{ */
{
	char *found;

	found = php_memnstr(haystack, needle, needle_len, haystack + haystack_len);
	if ((found - haystack) == 0) 
	{
		return 1;
	}
	else
	{
		return 0;
	}
}
/* }}} */

static int mmc_sendcmd(mmc_t *mmc, const char *cmd, int cmdlen TSRMLS_DC) /* {{{ */
{
	char *command;
	int   command_len;

	if (!mmc || !cmd) 
	{
		return -1;
	}

	MMC_DEBUG(("mmc_sendcmd: sending command '%s'", cmd));

	command 	= emalloc(cmdlen + sizeof("\r\n"));
	memcpy(command, cmd, cmdlen);
	memcpy(command + cmdlen, "\r\n", sizeof("\r\n") - 1);
	command_len = cmdlen + sizeof("\r\n") - 1;
	command[command_len] = '\0';

	if (php_stream_write(mmc->stream, command, command_len) != command_len) 
	{
		MMC_DEBUG(("mmc_sendcmd: write failed"));
		efree(command);
		return -1;
	}
	efree(command);

	return 1;
}
/* }}}*/

static int mmc_exec_storage_cmd(mmc_t *mmc, char *command, int command_len, char *key, int key_len, int queue_len, int flags, int expire, char *dep, int dep_len, char *data, int data_len TSRMLS_DC) /* {{{ */
{
	char *	real_command;
	int 	size;
	int		response_buf_size;

	if (strcmp(command, "queue") == 0)
	{
		real_command = emalloc(	  command_len
								+ 1				/* space */
								+ key_len
								+ 1				/* space */
								+ MAX_LENGTH_OF_LONG
								+ 1 			/* space */
								+ MAX_LENGTH_OF_LONG
								+ 1 			/* space */
								+ MAX_LENGTH_OF_LONG
								+ 1 			/* space */
								+ MAX_LENGTH_OF_LONG
								+ 1				/* space */
								+ dep_len
								+ sizeof("\r\n") - 1
								+ data_len
								+ sizeof("\r\n") - 1
								+ 1
								);
	
		size = sprintf(real_command, "%s %s %d %d %d %d", command, key, queue_len, flags, expire, data_len);
	}
	else
	{
		real_command = emalloc(	  command_len
								+ 1				/* space */
								+ key_len
								+ 1				/* space */
								+ MAX_LENGTH_OF_LONG
								+ 1 			/* space */
								+ MAX_LENGTH_OF_LONG
								+ 1 			/* space */
								+ MAX_LENGTH_OF_LONG
								+ 1				/* space */
								+ dep_len
								+ sizeof("\r\n") - 1
								+ data_len
								+ sizeof("\r\n") - 1
								+ 1
								);
	
		size = sprintf(real_command, "%s %s %d %d %d", command, key, flags, expire, data_len);
	}
	
	if (dep_len > 0)
	{
		memcpy(real_command + size, " ", 1);
		size += 1;
		
		memcpy(real_command + size, dep, dep_len);
		size += dep_len;
	}
	memcpy(real_command + size, "\r\n", sizeof("\r\n") - 1);
	size += sizeof("\r\n") - 1;
	memcpy(real_command + size, data, data_len);
	size += data_len;
	memcpy(real_command + size, "\r\n", sizeof("\r\n") - 1);
	size += sizeof("\r\n") - 1;
	real_command[size] = '\0';

	MMC_DEBUG(("mmc_exec_storage_cmd: store cmd is '%s'", real_command));
	MMC_DEBUG(("mmc_exec_storage_cmd: trying to store '%s', %d bytes", data, data_len));

	/* send command & data */
	if (php_stream_write(mmc->stream, real_command, size) != size) 
	{
		MMC_DEBUG(("failed to send command and data to the server"));
		efree(real_command);
		return -1;
	}
	efree(real_command);

	/* get server's response */
	response_buf_size = mmc_readline(mmc TSRMLS_CC);
	if (response_buf_size < 0) 
	{
		MMC_DEBUG(("failed to read the server's response"));
		return -1;
	}

	MMC_DEBUG(("mmc_exec_storage_cmd: response is '%s'", mmc->inbuf));

	/* stored or not? */
	if (mmc_str_left(mmc->inbuf,"STORED", response_buf_size, sizeof("STORED") - 1)) 
	{
		return 1;
	}

	/* not stored, return FALSE */
	if (mmc_str_left(mmc->inbuf,"NOT_STORED", response_buf_size, sizeof("NOT_STORED") - 1)) 
	{
		return 0;
	}

	/* return FALSE without failover */
	if (mmc_str_left(mmc->inbuf, "SERVER_ERROR out of memory", response_buf_size, sizeof("SERVER_ERROR out of memory") - 1)) 
	{
		return 0;
	}

	MMC_DEBUG(("an error occured while trying to store the item on the server"));
	return -1;
}
/* }}} */

static int mmc_parse_response(char *response, char **item_name, int response_len, int *flags, int *value_len) /* {{{ */
{
	int i = 0;
	int n = 0;
	int spaces[3];

	if (!response || response_len <= 0) 
	{
		return -1;
	}

	MMC_DEBUG(("mmc_parse_response: got response '%s'", response));

	for (i = 0; i < response_len; i++) 
	{
		if (response[i] == ' ') 
		{
			spaces[n] = i;
			n++;
			if (n == 3) 
			{
				break;
			}
		}
	}

	MMC_DEBUG(("mmc_parse_response: found %d spaces", n));

	if (n < 3) 
	{
		return -1;
	}

	if (item_name != NULL) 
	{
		int item_name_len;
		
		item_name_len = spaces[1] - spaces[0] - 1;
		*item_name    = emalloc(item_name_len + 1);
		memcpy(*item_name, response + spaces[0] + 1, item_name_len);
		(*item_name)[item_name_len] = '\0';
	}

	*flags     = atoi(response + spaces[1]);
	*value_len = atoi(response + spaces[2]);

	if (*flags < 0 || *value_len < 0) 
	{
		return -1;
	}

	MMC_DEBUG(("mmc_parse_response: 1st space is at %d position", spaces[1]));
	MMC_DEBUG(("mmc_parse_response: 2nd space is at %d position", spaces[2]));
	MMC_DEBUG(("mmc_parse_response: flags = %d", *flags));
	MMC_DEBUG(("mmc_parse_response: value_len = %d ", *value_len));

	return 1;
}
/* }}} */

/*
 * Retrieval might always return a different number of items.
 * Less items, because not all items were cached
 * More items, because one or more of the items was a queue
 * 
 * With any storage command we always send the command off and then accept whatever is sent
 * back to us.  Then we first collect all queued items together and then add all other items.
 */
 
 
static int mmc_exec_retrieval_cmd(char *cmd, mmc_pool_t *pool, zval *key, zval **return_value TSRMLS_DC) /* {{{ */
{
	mmc_t *	mmc;
	char *	request;
	int		result;
	int		request_len;
	int		response_len;
	int		retry;
	int		has_value;
	
	result    = -1;
	has_value = 0;
	
	MMC_DEBUG(("mmc_exec_retrieval_cmd: key '%s'", Z_STRVAL_P(key)));

	convert_to_string(key);
	MMC_PREPARE_KEY(Z_STRVAL_P(key), Z_STRLEN_P(key));

	/* get + ' ' + key + \0 */
	request     = emalloc(strlen(cmd) + Z_STRLEN_P(key) + 2);
	request_len = sprintf(request, "%s %s", cmd, Z_STRVAL_P(key));

	while (result < 0 && (mmc = mmc_server_find(pool, Z_STRVAL_P(key), Z_STRLEN_P(key) TSRMLS_CC)) != NULL) 
	{
		result = mmc_sendcmd(mmc, request, request_len TSRMLS_CC);
		if (result > 0)
		{
			mmc_key_value_t	*	keyval;
			mmc_key_value_t	*	k;
			int					first;
	
			MMC_DEBUG(("mmc_exec_retrieval_cmd: found server '%s:%d' for key '%s'", mmc->host, mmc->port, Z_STRVAL_P(key)));
			
			keyval = NULL;
			mmc_read_all_values(mmc, &keyval TSRMLS_CC);
			
			/* save the first returned value to the return_value */
			first = 1;
			if (keyval != NULL)
			{
				while (keyval != NULL)
				{
					if (first)
					{
						ZVAL_ZVAL(*return_value, keyval->value, 1, 1);
						first		  = 0;
						has_value	  = 1;
					}
					else
					{
						zval_ptr_dtor(&keyval->value);
					}
					k 	   = keyval;
					keyval = keyval->next;
					efree(k->key);
					efree(k);
				}
			}
		}

		if (result < 0 && mmc_server_failure(mmc TSRMLS_CC))
		{
			php_error_docref(NULL TSRMLS_CC, E_NOTICE, "marked server '%s:%d' as failed", mmc->host, mmc->port);
		}
	}

	if (!has_value)
	{
		ZVAL_FALSE(*return_value);
	}

	efree(request);
	return result;
}
/* }}} */

static int mmc_exec_retrieval_cmd_multi(char *cmd, mmc_pool_t *pool, zval *keys, zval **return_value TSRMLS_DC) /* {{{ */
{
	mmc_t *			mmc;
	HashPosition 	pos;
	zval **			key;
	zval *			value;
	char *			result_key;
	int				i = 0; 
	int				j;
	int				num_requests;
	int				result;
	int				result_status;

	array_init(*return_value);

	/* Until no retrieval errors or all servers have failed */
	do 
	{
		result_status = 0;
		num_requests  = 0;
		zend_hash_internal_pointer_reset_ex(Z_ARRVAL_P(keys), &pos);

		/* first pass to build requests for each server */
		while (zend_hash_get_current_data_ex(Z_ARRVAL_P(keys), (void **) &key, &pos) == SUCCESS) 
		{
			if (Z_TYPE_PP(key) != IS_STRING) 
			{
				SEPARATE_ZVAL(key);
				convert_to_string(*key);
			}

			MMC_PREPARE_KEY(Z_STRVAL_PP(key), Z_STRLEN_PP(key));

			/* schedule key if first round or if missing from result */
			if (	(	!i 
					||	!zend_hash_exists(Z_ARRVAL_PP(return_value), Z_STRVAL_PP(key), Z_STRLEN_PP(key))) 
				&&	(mmc = mmc_server_find(pool, Z_STRVAL_PP(key), Z_STRLEN_PP(key) TSRMLS_CC)) != NULL) 
			{
				if (!(mmc->outbuf.len)) 
				{
					smart_str_appendl(&(mmc->outbuf), cmd, strlen(cmd));
					pool->requests[num_requests++] = mmc;
				}

				smart_str_appendl(&(mmc->outbuf), " ", 1);
				smart_str_appendl(&(mmc->outbuf), Z_STRVAL_PP(key), Z_STRLEN_PP(key));
				MMC_DEBUG(("mmc_exec_retrieval_cmd_multi: scheduled key '%s' for '%s:%d' request length '%d'", Z_STRVAL_PP(key), mmc->host, mmc->port, mmc->outbuf.len));
			}

			zend_hash_move_forward_ex(Z_ARRVAL_P(keys), &pos);
		}

		/* second pass to send requests in parallel */
		for (j=0; j<num_requests; j++)
		{
			smart_str_0(&(pool->requests[j]->outbuf));
			result = mmc_sendcmd(pool->requests[j], pool->requests[j]->outbuf.c, pool->requests[j]->outbuf.len TSRMLS_CC);
			if (result < 0) 
			{
				if (mmc_server_failure(pool->requests[j] TSRMLS_CC)) 
				{
					php_error_docref(NULL TSRMLS_CC, E_NOTICE, "marked server '%s:%d' as failed", pool->requests[j]->host, pool->requests[j]->port);
				}
				result_status = result;
			}
		}

		/* third pass to read responses */
		for (j=0; j<num_requests; j++) 
		{
			if (pool->requests[j]->status != MMC_STATUS_FAILED) 
			{
				mmc_key_value_t	* keyval;
				mmc_key_value_t	* k;

				keyval = NULL;
				result = mmc_read_all_values(pool->requests[j], &keyval TSRMLS_CC);
				
				/* return all fetched values */
				while (keyval != NULL)
				{
					add_assoc_zval(*return_value, keyval->key, keyval->value);

					k 	   = keyval;
					keyval = keyval->next;
					efree(k->key);
					efree(k);
				}
				
				/* check for server failure */
				if (result < 0) 
				{
					if (mmc_server_failure(pool->requests[j] TSRMLS_CC)) 
					{
						php_error_docref(NULL TSRMLS_CC, E_NOTICE, "marked server '%s:%d' as failed", pool->requests[j]->host, pool->requests[j]->port);
					}
					result_status = result;
				}
			}

			smart_str_free(&(pool->requests[j]->outbuf));
		}
	}
	while (result_status < 0 && i++ < 20);

	return result_status;
}
/* }}} */


/* Fetch all items, return a linked list of items. Items within the same queue are linked into
 * the value list.
 */
static int mmc_read_all_values ( mmc_t *mmc, mmc_key_value_t **vals TSRMLS_DC )
{
	zval *				value;
	char *				key;
	mmc_key_value_t	*	keyval;
	char *				qhash;
	char 				idx[20];
	int					ret;
		
	keyval = NULL;
	*vals  = NULL;
	do
	{
		key   = NULL;
		value = NULL;
		ret   = mmc_read_value(mmc, &key, &value TSRMLS_CC);
		if (ret == 1)
		{
			 qhash = strstr(key, "##");
			 if (qhash)
			 {
			 	/* Queued item, create an array with the queued values */
			 	*qhash = '\0';
			 	if (!keyval || strcmp(key, keyval->key) != 0)
			 	{
					/* New item needed */
				 	if (keyval == NULL)
				 	{
				 		keyval			= emalloc(sizeof(mmc_key_value_t));
				 		*vals			= keyval;
				 	}
				 	else
				 	{
				 		keyval->next	= emalloc(sizeof(mmc_key_value_t));
				 		keyval			= keyval->next;
				 	}
					keyval->next		= NULL;
			 		keyval->key   		= key;
			 		keyval->queue_count = 0;
					key					= NULL;

					/* The array that will contain all values */
					MAKE_STD_ZVAL(keyval->value);
					array_init(keyval->value);
			 	}
				
				/* Append to current item */
				sprintf(idx, "%d", keyval->queue_count);
				add_assoc_zval(keyval->value, idx, value);
				keyval->queue_count += 1;
			 }
			 else
			 {
			 	/* Normal value returned, add to the queue of items */
			 	if (keyval)
			 	{
			 		keyval->next	= emalloc(sizeof(mmc_key_value_t));
			 		keyval			= keyval->next;
			 	}
			 	else
			 	{
			 		keyval			= emalloc(sizeof(mmc_key_value_t));
			 		*vals			= keyval;
			 	}
				keyval->next		= NULL;
		 		keyval->key			= key;
		 		keyval->value		= value;
		 		keyval->queue_count = 0;
		 		key 				= NULL;
			 }
		}

		if (key)
		{
			efree(key);
		}
	}
	while (ret == 1);
	return ret;
}

 


/**
 * Read a value from a depcached server get-reply stream.
 * The returned value could be 'END', then return 0
 * When it is a value return 1
 * On an error return -1.
 * 
 * @param mmc_t mmc		the server connection
 * @param char ** key	set to the varname read (if return 0)
 * @param zval ** value	set to the value read (if return 0)
 * @return int 
 */ 
static int mmc_read_value ( mmc_t *mmc, char **key, zval **value TSRMLS_DC ) /* {{{ */
{
	int response_len, flags, data_len, i, size;
	char *data;

	/* read "VALUE <key> <flags> <bytes>\r\n" header line */
	if ((response_len = mmc_readline(mmc TSRMLS_CC)) < 0) 
	{
		MMC_DEBUG(("failed to read the server's response"));
		return -1;
	}

	/* reached the end of the data */
	if (mmc_str_left(mmc->inbuf, "END", response_len, sizeof("END") - 1)) 
	{
		return 0;
	}

	if (mmc_parse_response(mmc->inbuf, key, response_len, &flags, &data_len) < 0) 
	{
		return -1;
	}

	MMC_DEBUG(("mmc_read_value: data len is %d bytes", data_len));

	/* data_len + \r\n + \0 */
	data = emalloc(data_len + 3);

	for (i=0; i<data_len+2; i+=size) 
	{
		if ((size = php_stream_read(mmc->stream, data + i, data_len + 2 - i)) == 0) 
		{
			MMC_DEBUG(("incomplete data block (expected %d, got %d)", (data_len + 2), i));
			if (key) 
			{
				efree(*key);
			}
			efree(data);
			return -1;
		}
	}

	data[data_len] = '\0';
	if (data_len == 0)
	{
		if (*value == NULL) 
		{
			MAKE_STD_ZVAL(*value);
		}
		ZVAL_EMPTY_STRING(*value);
		efree(data);
		return 1;
	}

	if (flags & MMC_COMPRESSED) 
	{
		char *result_data;
		long  result_len = 0;

		if (!mmc_uncompress(&result_data, &result_len, data, data_len))
		{
			MMC_DEBUG(("Error when uncompressing data"));
			if (key) 
			{
				efree(*key);
			}
			efree(data);
			return -1;
		}

		efree(data);
		data = result_data;
		data_len = result_len;
	}

	MMC_DEBUG(("mmc_read_value: data '%s'", data));

	if (*value == NULL) 
	{
		MAKE_STD_ZVAL(*value);
	}

	if (flags & MMC_SERIALIZED) 
	{
		const char *			tmp = data;
		php_unserialize_data_t 	var_hash;
		PHP_VAR_UNSERIALIZE_INIT(var_hash);

		if (!php_var_unserialize(value, (const unsigned char **)&tmp, (const unsigned char *) tmp + data_len, &var_hash TSRMLS_CC)) 
		{
			MMC_DEBUG(("Error at offset %d of %d bytes", tmp - data, data_len));
			PHP_VAR_UNSERIALIZE_DESTROY(var_hash);
			if (key) 
			{
				efree(*key);
			}
			efree(data);
			return -1;
		}

		PHP_VAR_UNSERIALIZE_DESTROY(var_hash);
		efree(data);
	}
	else 
	{
		ZVAL_STRINGL(*value, data, data_len, 0);
	}

	return 1;
}
/* }}} */

static int mmc_delete(mmc_t *mmc, char *key, int key_len, int time TSRMLS_DC) /* {{{ */
{
	char *	real_command;
	int 	size;
	int		response_buf_size;

	real_command = emalloc(	  sizeof("delete") - 1
							+ 1						/* space */
							+ key_len
							+ 1						/* space */
							+ MAX_LENGTH_OF_LONG
							+ 1
							);

	size = sprintf(real_command, "delete %s %d", key, time);

	MMC_DEBUG(("mmc_delete: trying to delete '%s'", key));

	/* drop it! =) */
	if (mmc_sendcmd(mmc, real_command, size TSRMLS_CC) < 0) 
	{
		efree(real_command);
		return -1;
	}
	efree(real_command);

	/* get server's response */
	response_buf_size = mmc_readline(mmc TSRMLS_CC);
	if (response_buf_size < 0)
	{
		MMC_DEBUG(("failed to read the server's response"));
		return -1;
	}

	MMC_DEBUG(("mmc_delete: server's response is '%s'", mmc->inbuf));

	/* ok? */
	if(mmc_str_left(mmc->inbuf,"DELETED", response_buf_size, sizeof("DELETED") - 1)) 
	{
		return 1;
	}

	if(mmc_str_left(mmc->inbuf,"NOT_FOUND", response_buf_size, sizeof("NOT_FOUND") - 1)) 
	{
		/* return 0, if such wasn't found */
		return 0;
	}

	MMC_DEBUG(("failed to delete item"));

	/* hmm.. */
	return -1;
}
/* }}} */


/**
 * Flush all keys on a server
 */
static int mmc_flush_all ( mmc_t *mmc TSRMLS_DC ) /* {{{ */
{
	int response_buf_size;

	MMC_DEBUG(("mmc_flush_all: flushing the cache"));

	if (mmc_sendcmd(mmc, "flush_all", sizeof("flush_all") - 1 TSRMLS_CC) < 0) 
	{
		return -1;
	}

	/* get server's response */
	response_buf_size = mmc_readline(mmc TSRMLS_CC);
	if (response_buf_size < 0)
	{
		return -1;
	}

	MMC_DEBUG(("mmc_flush_all: server's response is '%s'", mmc->inbuf));

	/* ok? */
	if(mmc_str_left(mmc->inbuf,"OK", response_buf_size, sizeof("OK") - 1)) 
	{
		return 1;
	}

	MMC_DEBUG(("failed to flush server's cache"));

	/* hmm.. */
	return -1;
}
/* }}} */


/**
 * Flush a specific dependency key
 */
static int mmc_flush ( mmc_t *mmc, char *key, int key_len TSRMLS_DC ) /* {{{ */
{
	char *	real_command;
	int 	size;
	int		response_buf_size;

	real_command = emalloc(	  sizeof("flush") - 1
							+ 1						/* space */
							+ 1000
							+ key_len
							+ 1
							);

	size = sprintf(real_command, "flush %*.*s", key_len, key_len, key);

	MMC_DEBUG(("mmc_flush: trying to flush dir '%s'", key));

	/* drop it! =) */
	if (mmc_sendcmd(mmc, real_command, size TSRMLS_CC) < 0) 
	{
		efree(real_command);
		return -1;
	}
	efree(real_command);

	/* get server's response */
	if ((response_buf_size = mmc_readline(mmc TSRMLS_CC)) < 0)
	{
		MMC_DEBUG(("failed to read the server's response"));
		return -1;
	}

	MMC_DEBUG(("mmc_flush: server's response is '%s'", mmc->inbuf));

	/* ok? */
	if (mmc_str_left(mmc->inbuf,"OK", response_buf_size, sizeof("OK") - 1)) 
	{
		return 1;
	}

	MMC_DEBUG(("failed to flush dependency"));

	/* hmm.. */
	return -1;
}
/* }}} */


static void php_mmc_store (INTERNAL_FUNCTION_PARAMETERS, char *command, int command_len) /* {{{ */
{
	mmc_t *		mmc;
	mmc_pool_t *pool;
	int 		result		= -1;
	int			value_len;
	int			data_len;
	char *		value;
	char *		data;
	char *		key;
	int			key_len;
	int			queue_len	= 0;
	char *		real_key;
	long 		flags		= 0;
	long		expire 		= 0;
	char *		dep;
	int			dep_len		= 0;
	zval *		var;
	zval *		mmc_object	= getThis();
	php_serialize_data_t var_hash;
	smart_str 	buf			= {0};

	/* TODO: handle special queue command, different parameters */
	if (mmc_object == NULL) 
	{
		if (strcmp(command, "queue") == 0)
		{
			if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "Oszl|lls", &mmc_object, depcached_class_entry_ptr, &key, &key_len, &var, &queue_len, &flags, &expire, &dep, &dep_len) == FAILURE) 
			{
				return;
			}
		}
		else
		{
			if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "Osz|lls", &mmc_object, depcached_class_entry_ptr, &key, &key_len, &var, &flags, &expire, &dep, &dep_len) == FAILURE) 
			{
				return;
			}
		}
	}
	else 
	{
		if (strcmp(command, "queue") == 0)
		{
			if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "szl|lls", &key, &key_len, &var, &queue_len, &flags, &expire, &dep, &dep_len) == FAILURE) 
			{
				return;
			}
		}
		else 
		{
			if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "sz|lls", &key, &key_len, &var, &flags, &expire, &dep, &dep_len) == FAILURE) 
			{
				return;
			}
		}
	}

	if (!mmc_get_pool(mmc_object, &pool TSRMLS_CC) || !pool->num_servers) 
	{
		RETURN_FALSE;
	}

	MMC_PREPARE_KEY(key, key_len);

	if (key_len > MMC_KEY_MAX_SIZE) 
	{
		real_key = estrndup(key, MMC_KEY_MAX_SIZE);
		key_len  = MMC_KEY_MAX_SIZE;
	}
	else 
	{
		real_key = estrdup(key);
	}

	switch (Z_TYPE_P(var)) 
	{
	case IS_STRING:
		value     = Z_STRVAL_P(var);
		value_len = Z_STRLEN_P(var);
		break;

	case IS_LONG:
	case IS_DOUBLE:
	case IS_BOOL:
		convert_to_string(var);
		value     = Z_STRVAL_P(var);
		value_len = Z_STRLEN_P(var);
		break;

	default:
		PHP_VAR_SERIALIZE_INIT(var_hash);
		php_var_serialize(&buf, &var, &var_hash TSRMLS_CC);
		PHP_VAR_SERIALIZE_DESTROY(var_hash);

		if (!buf.c) 
		{
			/* you're trying to save null or something went really wrong */
			RETURN_FALSE;
		}

		value     = buf.c;
		value_len = buf.len;
		flags    |= MMC_SERIALIZED;
		break;
	}

	/* autocompress large values */
	if (pool->compress_threshold && value_len >= pool->compress_threshold) 
	{
		flags |= MMC_COMPRESSED;
	}

	if (flags & MMC_COMPRESSED) 
	{
		if (!mmc_compress(&data, &data_len, value, value_len TSRMLS_CC)) 
		{
			RETURN_FALSE;
		}

		MMC_DEBUG(("php_mmc_store: compressed '%s' from %d bytes to %d", key, value_len, data_len));

		/* was enough space was saved to motivate uncompress processing on get() */
		if (data_len >= value_len * (1 - pool->min_compress_savings)) 
		{
			efree(data);
			data	  = value;
			data_len  = value_len;
			flags 	 &= ~MMC_COMPRESSED;
			MMC_DEBUG(("php_mmc_store: compression saving were less that '%f', clearing compressed flag", pool->min_compress_savings));
		}
	}
	else 
	{
		data 	 = value;
		data_len = value_len;
	}

	while (result < 0 && (mmc = mmc_server_find(pool, real_key, key_len TSRMLS_CC)) != NULL)
	{
		result = mmc_exec_storage_cmd(mmc, command, command_len, real_key, key_len, queue_len, flags, expire, dep, dep_len, data, data_len TSRMLS_CC);
		if (result < 0 && mmc_server_failure(mmc TSRMLS_CC)) 
		{
			php_error_docref(NULL TSRMLS_CC, E_NOTICE, "marked server '%s:%d' as failed", mmc->host, mmc->port);
		}
	}

	if (flags & MMC_SERIALIZED) 
	{
		smart_str_free(&buf);
	}
	if (flags & MMC_COMPRESSED) 
	{
		efree(data);
	}
	efree(real_key);

	if (result > 0) 
	{
		RETURN_TRUE;
	}
	RETURN_FALSE;
}
/* }}} */


static void php_mmc_connect (INTERNAL_FUNCTION_PARAMETERS, int persistent) /* {{{ */
{
	zval *			mmc_object 	 = getThis();
	mmc_t *			mmc 		 = NULL;
	mmc_pool_t *	pool;
	int 			errnum 		 = 0;
	int				host_len;
	char *			host;
	char *			error_string = NULL;
	long 			port 		 = DEPCACHED_G(default_port);
	long			timeout		 = MMC_DEFAULT_TIMEOUT;

	if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "s|ll", &host, &host_len, &port, &timeout) == FAILURE) 
	{
		return;
	}

	/* initialize and connect server struct */
	if (persistent) 
	{
		mmc = mmc_find_persistent(host, host_len, port, timeout, MMC_DEFAULT_RETRY TSRMLS_CC);
	}
	else 
	{
		MMC_DEBUG(("php_mmc_connect: creating regular connection"));
		mmc = mmc_server_new(host, host_len, port, 0, timeout, MMC_DEFAULT_RETRY TSRMLS_CC);
	}

	if (!mmc_open(mmc, 1, &error_string, &errnum TSRMLS_CC)) 
	{
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Can't connect to %s:%ld, %s (%d)", host, port, error_string ? error_string : "Unknown error", errnum);
		if (error_string) 
		{
			efree(error_string);
		}
		RETURN_FALSE;
	}

	/* initialize pool */
	MMC_DEBUG(("php_mmc_connect: initializing server pool"));
	pool     = mmc_pool_new();
	pool->id = zend_list_insert(pool, le_depcached_pool);
	mmc_pool_add(pool, mmc, 1);

	if (mmc_object == NULL) 
	{
		object_init_ex(return_value, depcached_class_entry_ptr);
		add_property_resource(return_value, "connection", pool->id);
	}
	else 
	{
		add_property_resource(mmc_object, "connection", pool->id);
		RETURN_TRUE;
	}
}
/* }}} */

/* ----------------
   module functions
   ---------------- */

/* {{{ proto object depcached_connect( string host [, int port [, int timeout ] ])
   Connects to server and returns Demcache object */
PHP_FUNCTION(depcached_connect)
{
	php_mmc_connect(INTERNAL_FUNCTION_PARAM_PASSTHRU, 0);
}
/* }}} */

/* {{{ proto object depcached_pconnect( string host [, int port [, int timeout ] ])
   Connects to server and returns Depcached object */
PHP_FUNCTION(depcached_pconnect)
{
	php_mmc_connect(INTERNAL_FUNCTION_PARAM_PASSTHRU, 1);
}
/* }}} */

/* {{{ proto bool depcached_add_server( string host [, int port [, bool persistent [, int weight [, int timeout [, int retry_interval ] ] ] ] ])
   Adds a connection to the pool. The order in which this function is called is significant */
PHP_FUNCTION(depcached_add_server)
{
	zval **		connection;
	zval *		mmc_object 		= getThis();
	mmc_pool_t *pool;
	mmc_t *		mmc;
	long 		port 			= DEPCACHED_G(default_port);
	long		weight 			= 1;
	long		timeout 		= MMC_DEFAULT_TIMEOUT;
	long		retry_interval	= MMC_DEFAULT_RETRY;
	zend_bool 	persistent		= 1;
	int 		resource_type;
	int			host_len;
	char *		host;

	if (mmc_object) 
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "s|lblll", &host, &host_len, &port, &persistent, &weight, &timeout, &retry_interval) == FAILURE) 
		{
			return;
		}
	}
	else 
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "Os|lblll", &mmc_object, depcached_class_entry_ptr, &host, &host_len, &port, &persistent, &weight, &timeout, &retry_interval) == FAILURE) 
		{
			return;
		}
	}

	if (weight < 0) 
	{
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "weight must be a positive integer");
		RETURN_FALSE;
	}

	/* lazy initialization of server struct */
	if (persistent) 
	{
		mmc = mmc_find_persistent(host, host_len, port, timeout, retry_interval TSRMLS_CC);
	}
	else 
	{
		MMC_DEBUG(("depcached_add_server: initializing regular struct"));
		mmc = mmc_server_new(host, host_len, port, 0, timeout, retry_interval TSRMLS_CC);
	}

	/* initialize pool if need be */
	if (zend_hash_find(Z_OBJPROP_P(mmc_object), "connection", sizeof("connection"), (void **)&connection) == FAILURE) 
	{
		MMC_DEBUG(("mmc_add_connection: initialized and appended server to empty pool"));

		pool     = mmc_pool_new();
		pool->id = zend_list_insert(pool, le_depcached_pool);

		add_property_resource(mmc_object, "connection", pool->id);
	}
	else 
	{
		MMC_DEBUG(("depcached_add_server: appended server to pool"));

		pool = (mmc_pool_t *) zend_list_find(Z_LVAL_PP(connection), &resource_type);
		if (!pool || resource_type != le_depcached_pool) 
		{
			php_error_docref(NULL TSRMLS_CC, E_WARNING, "connection identifier not found");
			RETURN_FALSE;
		}
	}

	mmc_pool_add(pool, mmc, weight);
	RETURN_TRUE;
}
/* }}} */

static mmc_t *mmc_find_persistent(char *host, int host_len, int port, int timeout, int retry_interval TSRMLS_DC) /* {{{ */
{
	mmc_t *		mmc;
	list_entry *le;
	char *		hash_key;
	int 		hash_key_len;

	MMC_DEBUG(("mmc_find_persistent: seeking for persistent connection"));
	
	hash_key     = emalloc(sizeof("mmc_connect___") - 1 + host_len + MAX_LENGTH_OF_LONG + 1);
	hash_key_len = sprintf(hash_key, "mmc_connect___%s:%d", host, port);

	if (zend_hash_find(&EG(persistent_list), hash_key, hash_key_len+1, (void **) &le) == FAILURE) 
	{
		list_entry new_le;

		MMC_DEBUG(("mmc_find_persistent: connection wasn't found in the hash"));

		mmc			= mmc_server_new(host, host_len, port, 1, timeout, retry_interval TSRMLS_CC);
		new_le.type = le_pdepcached;
		new_le.ptr  = mmc;

		/* register new persistent connection */
		if (zend_hash_update(&EG(persistent_list), hash_key, hash_key_len+1, (void *) &new_le, sizeof(list_entry), NULL) == FAILURE) 
		{
			mmc_server_free(mmc TSRMLS_CC);
			mmc = NULL;
		}
		else
		{
			zend_list_insert(mmc, le_pdepcached);
		}
	}
	else if (le->type != le_pdepcached || le->ptr == NULL) 
	{
		list_entry new_le;
		
		MMC_DEBUG(("mmc_find_persistent: something was wrong, reconnecting.."));
		
		zend_hash_del(&EG(persistent_list), hash_key, hash_key_len+1);

		mmc 		= mmc_server_new(host, host_len, port, 1, timeout, retry_interval TSRMLS_CC);
		new_le.type = le_pdepcached;
		new_le.ptr  = mmc;

		/* register new persistent connection */
		if (zend_hash_update(&EG(persistent_list), hash_key, hash_key_len+1, (void *) &new_le, sizeof(list_entry), NULL) == FAILURE) 
		{
			mmc_server_free(mmc TSRMLS_CC);
			mmc = NULL;
		}
		else 
		{
			zend_list_insert(mmc, le_pdepcached);
		}
	}
	else 
	{
		MMC_DEBUG(("mmc_find_persistent: connection found in the hash"));

		mmc 				= (mmc_t *)le->ptr;
		mmc->timeout 		= timeout;
		mmc->retry_interval = retry_interval;

		/* attempt to reconnect this node before failover in case connection has gone away */
		if (mmc->status == MMC_STATUS_CONNECTED) 
		{
			mmc->status = MMC_STATUS_UNKNOWN;
		}
	}

	efree(hash_key);
	return mmc;
}
/* }}} */

/* {{{ proto string depcached_get_version( object depcached )
   Returns server's version */
PHP_FUNCTION(depcached_get_version)
{
	mmc_pool_t *pool;
	char *		version;
	int 		i;
	zval *		mmc_object = getThis();

	if (mmc_object == NULL) 
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "O", &mmc_object, depcached_class_entry_ptr) == FAILURE) 
		{
			return;
		}
	}

	if (!mmc_get_pool(mmc_object, &pool TSRMLS_CC))
	{
		RETURN_FALSE;
	}

	for (i=0; i<pool->num_servers; i++) 
	{
		if (	mmc_open(pool->servers[i], 1, NULL, NULL TSRMLS_CC)
			&&	(version = mmc_get_version(pool->servers[i] TSRMLS_CC)) != NULL) 
		{
			RETURN_STRING(version, 0);
		}
		else if (mmc_server_failure(pool->servers[i] TSRMLS_CC)) 
		{
			php_error_docref(NULL TSRMLS_CC, E_NOTICE, "marked server '%s:%d' as failed", pool->servers[i]->host, pool->servers[i]->port);
		}
	}

	RETURN_FALSE;
}
/* }}} */


/* {{{ proto bool depcached_set( object depcached, string key, mixed var [, int flag [, int expire ] ] )
   Sets the value of an item. Item may exist or not */
PHP_FUNCTION(depcached_set)
{
	php_mmc_store(INTERNAL_FUNCTION_PARAM_PASSTHRU, "set", sizeof("set") - 1);
}
/* }}} */


/* {{{ proto bool depcached_queue( object depcached, string key, mixed var, int queue_len, [, int flag [, int expire ] ] )
   Sets the value of an item. Item may exist or not */
PHP_FUNCTION(depcached_queue)
{
	php_mmc_store(INTERNAL_FUNCTION_PARAM_PASSTHRU, "queue", sizeof("queue") - 1);
}
/* }}} */


/* {{{ proto bool depcached_dequeue( object depcached, mixed key )
   Sets the value of an item. Item may exist or not */
PHP_FUNCTION(depcached_dequeue)
{
	mmc_pool_t *pool;
	zval *		key;
	zval *		mmc_object = getThis();

	if (mmc_object == NULL) 
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "Oz", &mmc_object, depcached_class_entry_ptr, &key) == FAILURE) 
		{
			return;
		}
	}
	else 
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "z", &key) == FAILURE) 
		{
			return;
		}
	}

	if (!mmc_get_pool(mmc_object, &pool TSRMLS_CC) || !pool->num_servers) 
	{
		RETURN_FALSE;
	}

	if (Z_TYPE_P(key) != IS_ARRAY) 
	{
		if (mmc_exec_retrieval_cmd("dequeue", pool, key, &return_value TSRMLS_CC) < 0) 
		{
			RETURN_FALSE;
		}
	}
	else 
	{
		if (mmc_exec_retrieval_cmd_multi("dequeue", pool, key, &return_value TSRMLS_CC) < 0) 
		{
			RETURN_FALSE;
		}
	}
}
/* }}} */


/* {{{ proto mixed depcached_get( object depcached, mixed key )
   Returns value of existing item or false */
PHP_FUNCTION(depcached_get)
{
	mmc_pool_t *pool;
	zval *		key;
	zval *		mmc_object = getThis();

	if (mmc_object == NULL) 
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "Oz", &mmc_object, depcached_class_entry_ptr, &key) == FAILURE) 
		{
			return;
		}
	}
	else 
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "z", &key) == FAILURE) 
		{
			return;
		}
	}

	if (!mmc_get_pool(mmc_object, &pool TSRMLS_CC) || !pool->num_servers) 
	{
		RETURN_FALSE;
	}

	if (Z_TYPE_P(key) != IS_ARRAY) 
	{
		if (mmc_exec_retrieval_cmd("get", pool, key, &return_value TSRMLS_CC) < 0) 
		{
			RETURN_FALSE;
		}
	}
	else 
	{
		if (mmc_exec_retrieval_cmd_multi("get", pool, key, &return_value TSRMLS_CC) < 0) 
		{
			RETURN_FALSE;
		}
	}
}
/* }}} */

/* {{{ proto bool depcached_delete( object depcached, string key [, int expire ])
   Deletes existing item */
PHP_FUNCTION(depcached_delete)
{
	mmc_t *mmc;
	mmc_pool_t *pool;
	int result = -1, key_len;
	zval *mmc_object = getThis();
	char *key;
	long time = 0;

	if (mmc_object == NULL) 
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "Os|l", &mmc_object, depcached_class_entry_ptr, &key, &key_len, &time) == FAILURE) 
		{
			return;
		}
	}
	else 
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "s|l", &key, &key_len, &time) == FAILURE) 
		{
			return;
		}
	}

	if (!mmc_get_pool(mmc_object, &pool TSRMLS_CC) || !pool->num_servers) 
	{
		RETURN_FALSE;
	}

	MMC_PREPARE_KEY(key, key_len);

	while (result < 0 && (mmc = mmc_server_find(pool, key, key_len TSRMLS_CC)) != NULL) 
	{
		result = mmc_delete(mmc, key, key_len, time TSRMLS_CC);
		if (result < 0 && mmc_server_failure(mmc TSRMLS_CC)) 
		{
			php_error_docref(NULL TSRMLS_CC, E_NOTICE, "marked server '%s:%d' as failed", mmc->host, mmc->port);
		}
	}

	if (result > 0)
	{
		RETURN_TRUE;
	}
	RETURN_FALSE;
}
/* }}} */

/* {{{ proto bool depcached_debug( bool onoff )
   Turns on/off internal debugging */
PHP_FUNCTION(depcached_debug)
{
#if ZEND_DEBUG
	zend_bool onoff;

	if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "b", &onoff) == FAILURE) 
	{
		return;
	}

	DEPCACHED_G(debug_mode) = onoff ? 1 : 0;

	RETURN_TRUE;
#else
	RETURN_FALSE;
#endif
}
/* }}} */


/* {{{ proto array depcached_set_compress_threshold( object depcached, int threshold [, float min_savings ] )
   Set automatic compress threshold */
PHP_FUNCTION(depcached_set_compress_threshold)
{
	mmc_pool_t *pool;
	zval *		mmc_object 	= getThis();
	long 		threshold;
	double 		min_savings = MMC_DEFAULT_SAVINGS;

	if (mmc_object == NULL) 
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "Ol|d", &mmc_object, depcached_class_entry_ptr, &threshold, &min_savings) == FAILURE)
		{
			return;
		}
	}
	else 
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "l|d", &threshold, &min_savings) == FAILURE) 
		{
			return;
		}
	}

	if (!mmc_get_pool(mmc_object, &pool TSRMLS_CC)) 
	{
		RETURN_FALSE;
	}

	if (threshold < 0) 
	{
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "threshold must be a positive integer");
		RETURN_FALSE;
	}
	pool->compress_threshold = threshold;

	if (min_savings != MMC_DEFAULT_SAVINGS) 
	{
		if (min_savings < 0 || min_savings > 1) 
		{
			php_error_docref(NULL TSRMLS_CC, E_WARNING, "min_savings must be a float in the 0..1 range");
			RETURN_FALSE;
		}
		pool->min_compress_savings = min_savings;
	}
	else
	{
		pool->min_compress_savings = MMC_DEFAULT_SAVINGS;
	}

	RETURN_TRUE;
}
/* }}} */


/* {{{ proto bool depcached_close( object depcached )
   Closes connection to depcached */
PHP_FUNCTION(depcached_close)
{
	mmc_pool_t *pool;
	int			i;
	int			failures   = 0;
	zval *		mmc_object = getThis();

	if (mmc_object == NULL)
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "O", &mmc_object, depcached_class_entry_ptr) == FAILURE) 
		{
			return;
		}
	}

	if (!mmc_get_pool(mmc_object, &pool TSRMLS_CC)) 
	{
		RETURN_FALSE;
	}

	for (i=0; i<pool->num_servers; i++) 
	{
		if (mmc_close(pool->servers[i] TSRMLS_CC) < 0) 
		{
			if (mmc_server_failure(pool->servers[i] TSRMLS_CC)) 
			{
				php_error_docref(NULL TSRMLS_CC, E_NOTICE, "marked server '%s:%d' as failed", pool->servers[i]->host, pool->servers[i]->port);
			}
			failures++;
		}
	}

	if (failures && failures >= pool->num_servers) 
	{
		RETURN_FALSE;
	}
	RETURN_TRUE;
}
/* }}} */

/* {{{ proto bool depcached_flush_all( object depcached)
   Flushes the cache */
PHP_FUNCTION(depcached_flush_all)
{
	mmc_pool_t *	pool;
	int 			i;
	int				failures   = 0;
	zval *			mmc_object = getThis();

	if (mmc_object == NULL) 
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "O", &mmc_object, depcached_class_entry_ptr) == FAILURE) 
		{
			return;
		}
	}

	if (!mmc_get_pool(mmc_object, &pool TSRMLS_CC)) 
	{
		RETURN_FALSE;
	}

	for (i=0; i<pool->num_servers; i++) 
	{
		if (!mmc_open(pool->servers[i], 1, NULL, NULL TSRMLS_CC) || mmc_flush_all(pool->servers[i] TSRMLS_CC) < 0) 
		{
			if (mmc_server_failure(pool->servers[i] TSRMLS_CC)) 
			{
				php_error_docref(NULL TSRMLS_CC, E_NOTICE, "marked server '%s:%d' as failed", pool->servers[i]->host, pool->servers[i]->port);
			}
			failures++;
		}
	}

	if (failures && failures >= pool->num_servers) 
	{
		RETURN_FALSE;
	}
	RETURN_TRUE;
}
/* }}} */


/* {{{ proto bool depcached_flush( object depcached, string key)
   Flushes a dependency key */
PHP_FUNCTION(depcached_flush)
{
	mmc_t *			mmc;
	mmc_pool_t *	pool;
	int 			result; 
	zval *			mmc_object;
	char *			key;
	int				key_len;
	int				i;
	int				failures;

	result 		= -1;
	mmc_object 	= getThis();
	failures	= 0;

	if (mmc_object == NULL) 
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "Os", &mmc_object, depcached_class_entry_ptr, &key, &key_len) == FAILURE) 
		{
			return;
		}
	}
	else 
	{
		if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "s", &key, &key_len) == FAILURE) 
		{
			return;
		}
	}

	if (!mmc_get_pool(mmc_object, &pool TSRMLS_CC) || !pool->num_servers) 
	{
		RETURN_FALSE;
	}

	MMC_PREPARE_KEY(key, key_len);

	for (i=0; i<pool->num_servers; i++) 
	{
		if (	!mmc_open(pool->servers[i], 1, NULL, NULL TSRMLS_CC) 
			||	mmc_flush(pool->servers[i], key, key_len TSRMLS_CC) < 0) 
		{
			if (mmc_server_failure(pool->servers[i] TSRMLS_CC)) 
			{
				php_error_docref(NULL TSRMLS_CC, E_NOTICE, "marked server '%s:%d' as failed", pool->servers[i]->host, pool->servers[i]->port);
			}
			failures++;
		}
	}

	if (failures && failures >= pool->num_servers) 
	{
		RETURN_FALSE;
	}
	RETURN_TRUE;
}
/* }}} */


#endif /* HAVE_DEPCACHED */
/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: noet sw=4 ts=4 fdm=marker
 * vim<600: noet sw=4 ts=4
 */
