/* -*- Mode: C; tab-width: 4; c-basic-offset: 4; indent-tabs-mode: nil -*- */
/*
 *  depcached - memory caching daemon with dependency checking
 *
 *  Copyright 2007 Mediamatic Lab.  All rights reserved.
 * 
 *  Authors:
 *      Marc Worrell <marc@mediamatic.nl>
 *
 *  Based on:
 * 
 *  memcached - memory caching daemon
 *
 *       http://www.danga.com/memcached/
 *
 *  Copyright 2003 Danga Interactive, Inc.  All rights reserved.
 *
 * 
 *  Use and distribution licensed under the BSD license.  See
 *  the LICENSE file for full text.
 *
 *  Authors:
 *      Anatoly Vorobey <mellon@pobox.com>
 *      Brad Fitzpatrick <brad@danga.com>
 *
 *  $Id: depcached.c 30245 2007-08-20 16:57:10Z marc $
 */

#include "config.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/signal.h>
#include <sys/resource.h>

/* some POSIX systems need the following definition
 * to get mlockall flags out of sys/mman.h.  */
#ifndef _P1003_1B_VISIBLE
#define _P1003_1B_VISIBLE
#endif

#include <pwd.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <errno.h>
#include <time.h>
#include <event.h>
#include <assert.h>

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#include "depcached.h"



enum conn_states {
    conn_listening,  /* the socket which listens for connections */
    conn_read,       /* reading in a command line */
    conn_write,      /* writing out a simple response */
    conn_nread,      /* reading in a fixed number of bytes */
    conn_swallow,    /* swallowing unnecessary bytes w/o storing */
    conn_closing,    /* closing this connection */
    conn_mwrite      /* writing out many items sequentially */
};


/* Commands that can be given to the cache */
enum comm {
	C_GET,
	C_SET,
	C_QUEUE,
	C_DEQUEUE,
	C_DELETE,
	C_FLUSH_ALL,
	C_FLUSH,
	C_STATS,
	C_VERSION,
	C_QUIT,
	C_ERROR
};


typedef struct {
    int		sfd;
    int		state;
    struct	event event;
    short	ev_flags;
    short	which;				/* which events were just triggered */

    char *	rbuf;  
    int		rsize;  
    int		rbytes;

    char *	wbuf;
    char *	wcurr;
    int		wsize;
    int		wbytes; 
    int		write_and_go;		/* which state to go into after finishing current write */
    void *	write_and_free;		/* free this memory after finishing writing */
    char	is_corked;			/* boolean, connection is corked */

    char *	rcurr;
    int		rlbytes;
	int		rqueue_len;
    char	rqueue_key[280];
    
    /* data for the nread state */

    /* 
     * item is used to hold an item structure created after reading the command
     * line of set/add/replace commands, but before we finished reading the actual 
     * data. The data is read into ITEM_data(item) to avoid extra copying.
     */

    t_item *	item;			/* for commands set/add/replace  */
    enum comm	item_comm;		/* which one is it: set/add/replace */

    /* data for the swallow state */
    int			sbytes;			/* how many bytes to swallow */

    /* data for the mwrite state */
    t_item **	ilist;			/* list of items to write out */
    int			isize;
    t_item **	icurr;
    int			ileft;
    int			ipart;			/* 1 if we're writing a VALUE line, 2 if we're writing data */
    char		ibuf[400];		/* for VALUE lines */
    char *		iptr;
    int			ibytes;
} conn;


/* 
 * given time value that's either unix time or delta from current unix time, return
 * unix time. Use the fact that delta can't exceed one month (and real time value can't 
 * be that low).
 */

time_t realtime(time_t exptime);


/* libevent handlers etc */
void	event_handler	( int fd, short which, void *arg );
void	drive_machine	( conn *c );
int		update_event	( conn *c, int new_flags );
int		try_read_command( conn *c );
int		try_read_network( conn *c );
void	complete_nread	( conn *c );
void	process_command	( conn *c, char *command );

/* connection */
static conn *	conn_new		( int sfd, int init_state, int event_flags );
static void		conn_close		( conn *c );
static void		conn_init		( void );
static int		new_socket		( void );
static int		server_socket	( int port );

/* stats */
static void		stats_reset		( void );
static void		stats_init		( void );

/* defaults */
static void		settings_init	( void );


/* variables */
struct stats	stats;
struct settings settings;


/* deletion queue with delayed deletes */
static t_item **todelete;
static int 		delcurr;
static int 		deltotal;

/* connection pool */
static conn **	freeconns;
static int 		freetotal;
static int 		freecurr;

/* the socket for listening */
static int 		l_socket = 0;

/* the current queue serial nr, incremented for every queue set */
static int 		queue_serial_nr = 0;

/* the current time at the start of the event */
time_t			now;


static struct {
	char *		cmd_s;
	enum comm	comm;
} cmds[] = {
	{ "get",		C_GET },
	{ "set",		C_SET },
	{ "queue",		C_QUEUE },
	{ "dequeue",	C_DEQUEUE },
	{ "delete",		C_DELETE },
	{ "flush_all",	C_FLUSH_ALL },
	{ "flush",		C_FLUSH },
	{ "stats",		C_STATS },
	{ "version",	C_VERSION },
	{ "quit",		C_QUIT },
	
	/* Just here for memcached compatibility */
	{ "add",		C_SET },
	{ "replace",	C_SET },
	{ NULL,			C_ERROR }
};


/**
 * Calculate the expiration time, when the exptime is smaller than 1 year then
 * take the exptime as a relative time.
 */
time_t realtime ( time_t exptime ) 
{
    /* no. of seconds in 1 year - largest possible delta exptime */
    #define REALTIME_MAXDELTA 60*60*24*365

    if (exptime == 0)
	{
		return 0; /* 0 means never expire */
	}

    if (exptime > REALTIME_MAXDELTA)
	{
        return exptime;
    }
	else 
	{
        return exptime + now;
    }
}


/**
 * Reset all statistics
 */
static void stats_init ( void ) 
{
	memset(&stats, 0, sizeof(stats));
    stats.started = time(0);
}


/** 
 * Reset some of the statistics
 */
static void stats_reset ( void ) 
{
    stats.total_items	= 0;
	stats.total_conns	= 0;
    stats.get_cmds		= 0;
	stats.set_cmds		= 0;
	stats.get_hits		= 0;
	stats.get_misses	= 0;
    stats.bytes_read	= 0;
	stats.bytes_written = 0;
}

/**
 * Initialise the settings struct to some default values
 */
static void settings_init ( void ) 
{
    settings.port				= 31307;
    settings.interface.s_addr	= htonl(INADDR_ANY);
    settings.maxbytes			= 64*1024*1024; /* default is 64MB */
    settings.maxconns			= 4096;         /* to limit connections-related memory to about 20MB */
    settings.verbose			= 0;
    settings.evict_to_free		= 1;       /* push old items out of cache when memory runs out */
}


static void set_cork ( conn *c, int val ) 
{
    if (c->is_corked != val)
	{
		c->is_corked = val;
#ifdef __APPLE_CC__
#undef TCP_NOPUSH
#endif
#ifdef TCP_NOPUSH
		setsockopt(c->sfd, IPPROTO_TCP, TCP_NOPUSH, &val, sizeof(val));
#endif
	}
}


/**
 * Initialise the free list of connections
 */
static void conn_init(void) 
{
    freetotal	= 200;
    freecurr	= 0;
    freeconns	= (conn **) calloc(freetotal, sizeof (conn *));
}


/**
 * Allocate a connection structure for a new client connection
 */
static conn *conn_new ( int sfd, int init_state, int event_flags ) 
{
    conn *c;

    /* Do we have a free conn structure from a previous close? */
    if (freecurr > 0) 
	{
        c = freeconns[--freecurr];
    }
	else
	{
		/* allocate a new connection */
        
		c = (conn *) malloc(sizeof(conn));
		if (!c) 
		{
            perror("malloc()");
            return NULL;
        }

        c->rbuf	 = (char *)    malloc(DATA_BUFFER_SIZE);
        c->wbuf	 = (char *)    malloc(DATA_BUFFER_SIZE);
        c->ilist = (t_item **) calloc(200, sizeof(t_item *));

        if (!c->rbuf || !c->wbuf || !c->ilist) 
		{
            if (c->rbuf)
			{
				free(c->rbuf);
            }
			if (c->wbuf)
			{
				free(c->wbuf);
            }
			if (c->ilist)
			{
				free(c->ilist);
            }
			free(c);
            perror("malloc()");
            return NULL;
        }
        c->rsize = DATA_BUFFER_SIZE;
		c->wsize = DATA_BUFFER_SIZE;
        c->isize = 200;
        stats.conn_structs++;
    }

    if (settings.verbose > 1) 
	{
        if (init_state == conn_listening)
		{
			fprintf(stderr, "<%d server listening\n", sfd);
        }
		else
		{
			fprintf(stderr, "<%d new client connection\n", sfd);
		}
	}

    c->sfd			= sfd;
    c->state		= init_state;
    c->rlbytes		= 0;
    c->rbytes		= 0;
	c->wbytes		= 0;
    c->wcurr		= c->wbuf;
    c->rcurr		= c->rbuf;
    c->icurr		= c->ilist; 
    c->ileft		= 0;
    c->iptr			= c->ibuf;
    c->ibytes		= 0;
    c->item			= 0;
    c->is_corked	= 0;
    c->write_and_go	= conn_read;
    c->write_and_free = 0;
    c->ev_flags		= event_flags;

	/* Let libevent handle the events for this new connection */
    event_set(&c->event, sfd, event_flags, event_handler, (void *)c);

    if (event_add(&c->event, 0) == -1) 
	{
        if (freecurr < freetotal) 
		{
            freeconns[freecurr++] = c;
        }
		else
		{
            free(c->rbuf);
            free(c->wbuf);
            free(c->ilist);
            free(c);
        }
        return NULL;
    }

    stats.curr_conns++;
    stats.total_conns++;

    return c;
}


/**
 * Close a client connection, removes the event handling for this connection.
 */
static void conn_close ( conn *c ) 
{
    /* delete the event, the socket and the conn */
    event_del(&c->event);

    if (settings.verbose > 1)
	{
        fprintf(stderr, "<%d connection closed.\n", c->sfd);
	}
	
    close(c->sfd);

    if (c->item) 
	{
        item_unref(c->item);
    }

    if (c->ileft) 
	{
        while (c->ileft > 0)
		{
			// TODO: use the item key to remove this item
            item_remove((*c->icurr)->h.key);
			c->ileft--;
			c->icurr++; 
        }
    }

    if (c->write_and_free) 
	{
        free(c->write_and_free);
    }

    /* if we have enough space in the free connections array, put the structure there */
    if (freecurr < freetotal) 
	{
        freeconns[freecurr++] = c;
    }
	else
	{
        /* delete the connection struct, we have enough free connections left */
		free(c->rbuf);
		free(c->wbuf);
		free(c->ilist);
		free(c);
    }
    stats.curr_conns--;
}



/**
 * echo a string to a client-connection 
 */
void out_string(conn *c, char *str) 
{
    int len;

    if (settings.verbose > 1)
	{
        fprintf(stderr, ">%d %s\n", c->sfd, str);
	}
	
    len = strlen(str);
    if (len + 2 >= c->wsize) 
	{
        /* ought to be always enough. just fail for simplicity */
        str = "SERVER_ERROR output line too long";
        len = strlen(str);
    }

    strcpy(c->wbuf, str);
    strcat(c->wbuf, "\r\n");
    c->wbytes		= len + 2;
    c->wcurr		= c->wbuf;
    c->state		= conn_write;
    c->write_and_go = conn_read;
}


/**
 * We get here after reading the value in the set command. The item is ready in c->item.
 */
void complete_nread ( conn *c ) 
{
    stats.set_cmds++;

    if (strncmp(c->item->data + c->item->datasz - 2, "\r\n", 2) != 0) 
	{
        out_string(c, "CLIENT_ERROR bad data chunk");
        item_unref(c->item);
    }
    else if (item_replace(c->item) != 0)
	{
		out_string(c, "NOT_STORED");
	    item_unref(c->item); 
	}
	else
	{
        /* Check if the item should be queue in a queue */
        if (c->rqueue_len > 0 && strlen(c->rqueue_key) > 0)
        {
        	if (!queue_item(c->rqueue_key, c->rqueue_len, c->item))
        	{
				out_string(c, "NOT_STORED");
			    item_unref(c->item); 
        	}
        	else
        	{
		        out_string(c, "STORED");
        	}
        }
        else
        {
	        out_string(c, "STORED");
        }
    }
    c->item = NULL;
}


/**
 * Queue a newly read item in an item queue
 */
int queue_item ( char *q_key, int q_len, t_item *item )
{
	t_item *q_item;
	int		i;
	
	q_item = item_find(q_key);
	if (!q_item || q_item->queue_len == 0)
	{
		/* queue item does not exist, or the older item is not a queue */
		q_item = item_alloc(q_key, "", item->locked, item->exptime, 0);
		item_replace(q_item);
		
		/* TODO: copy dependency keys from the item to the queue */
	}
	else
	{
		/* Use the latest expiry / lock state */
		/* When a queue is locked or set to never expire, then it always remains locked/never expire */
		q_item->locked	= (q_item->locked || item->locked);
		if (	item->exptime > q_item->exptime 
			&&	(q_item->exptime != 0 || q_item->exptime == 0))
		{
			q_item->exptime = item->exptime;
		}
	}

	if (q_item)
	{
		if (q_item->queue_len == 0)
		{
			/* allocate the item queue */
			q_item->queue_items = (t_item **) calloc(q_len, sizeof(t_item *));
			if (!q_item->queue_items)
			{
				item_remove(q_key);
				return 0;
			}

			q_item->queue_len = q_len;
		}
		else if (q_item->queue_len < q_len)
		{
			t_item **new_queue;
			
			/* reallocate the item queue */
			new_queue = (t_item **) realloc(q_item->queue_items, sizeof(t_item **) * q_len);
			if (!new_queue)
			{
				/* could not realloc */
				return 0;
			}

			for (i = q_item->queue_len; i < q_len; i++)
			{
				new_queue[i] = NULL;
			}
			q_item->queue_items = new_queue;
			q_item->queue_len   = q_len;
		}
		else if (q_item->queue_len > q_len)
		{
			/* queue shrinks, unref items above new high water mark */
			for (i = q_item->queue_len-1; i >= q_len; i--)
			{
				if (q_item->queue_items[i])
				{
					item_unref(q_item->queue_items[i]);
					q_item->queue_items[i] = NULL;
				}
			}
			q_item->queue_len = q_len;
		}
		
		/* place the new item in front of the queue */
		if (q_item->queue_items[q_item->queue_len-1])
		{
			item_unref(q_item->queue_items[q_item->queue_len-1]);
			q_item->queue_items[q_item->queue_len-1] = NULL;
		}
		
		for (i=q_item->queue_len-2; i>=0; i--)
		{
			q_item->queue_items[i+1] = q_item->queue_items[i];
		}
		q_item->queue_items[0] = item;
		item_ref(item);
		
		return 1;
	}
	else
	{
		return 0;
	}
}


/**
 * Return the different statistics.
 */
void process_stat ( conn *c, char *command ) 
{
    time_t now = time(0);

    if (strcmp(command, "stats") == 0) 
	{
        char temp[1024];
        pid_t pid = getpid();
        char *pos = temp;
        struct rusage usage;
        
        getrusage(RUSAGE_SELF, &usage);

        pos += sprintf(pos, "STAT pid %u\r\n", pid);
        pos += sprintf(pos, "STAT uptime %lu\r\n", now - stats.started);
        pos += sprintf(pos, "STAT time %ld\r\n", now);
        pos += sprintf(pos, "STAT version " VERSION "\r\n");
        pos += sprintf(pos, "STAT rusage_user %ld.%06ld\r\n", usage.ru_utime.tv_sec, (long) usage.ru_utime.tv_usec);
        pos += sprintf(pos, "STAT rusage_system %ld.%06ld\r\n", usage.ru_stime.tv_sec, (long) usage.ru_stime.tv_usec);
        pos += sprintf(pos, "STAT curr_items %u\r\n", stats.curr_items);
        pos += sprintf(pos, "STAT total_items %u\r\n", stats.total_items);
        pos += sprintf(pos, "STAT bytes %llu\r\n", stats.curr_bytes);
        pos += sprintf(pos, "STAT curr_connections %u\r\n", stats.curr_conns - 1); /* ignore listening conn */
        pos += sprintf(pos, "STAT total_connections %u\r\n", stats.total_conns);
        pos += sprintf(pos, "STAT connection_structures %u\r\n", stats.conn_structs);
        pos += sprintf(pos, "STAT cmd_get %u\r\n", stats.get_cmds);
        pos += sprintf(pos, "STAT cmd_set %u\r\n", stats.set_cmds);
        pos += sprintf(pos, "STAT get_hits %u\r\n", stats.get_hits);
        pos += sprintf(pos, "STAT get_misses %u\r\n", stats.get_misses);
        pos += sprintf(pos, "STAT bytes_read %llu\r\n", stats.bytes_read);
        pos += sprintf(pos, "STAT bytes_written %llu\r\n", stats.bytes_written);
        pos += sprintf(pos, "STAT limit_maxbytes %u\r\n", settings.maxbytes);
        pos += sprintf(pos, "END");
        out_string(c, temp);
        return;
    }

    if (strcmp(command, "stats reset") == 0) {
        stats_reset();
        out_string(c, "RESET");
        return;
    }

#ifdef HAVE_MALLOC_H
#ifdef HAVE_STRUCT_MALLINFO
    if (strcmp(command, "stats malloc") == 0) {
        char temp[512];
        struct mallinfo info;
        char *pos = temp;

        info = mallinfo();
        pos += sprintf(pos, "STAT arena_size %d\r\n", info.arena);
        pos += sprintf(pos, "STAT free_chunks %d\r\n", info.ordblks);
        pos += sprintf(pos, "STAT fastbin_blocks %d\r\n", info.smblks);
        pos += sprintf(pos, "STAT mmapped_regions %d\r\n", info.hblks);
        pos += sprintf(pos, "STAT mmapped_space %d\r\n", info.hblkhd);
        pos += sprintf(pos, "STAT max_total_alloc %d\r\n", info.usmblks);
        pos += sprintf(pos, "STAT fastbin_space %d\r\n", info.fsmblks);
        pos += sprintf(pos, "STAT total_alloc %d\r\n", info.uordblks);
        pos += sprintf(pos, "STAT total_free %d\r\n", info.fordblks);
        pos += sprintf(pos, "STAT releasable_space %d\r\nEND", info.keepcost);
        out_string(c, temp);
        return;
    }
#endif /* HAVE_STRUCT_MALLINFO */
#endif /* HAVE_MALLOC_H */

    if (strcmp(command, "stats maps") == 0) {
        char *wbuf;
        int wsize = 8192; /* should be enough */
        int fd;
        int res;

        wbuf = (char *)malloc(wsize);
        if (wbuf == 0) {
            out_string(c, "SERVER_ERROR out of memory");
            return;
        }
            
        fd = open("/proc/self/maps", O_RDONLY);
        if (fd == -1) {
            out_string(c, "SERVER_ERROR cannot open the maps file");
            free(wbuf);
            return;
        }

        res = read(fd, wbuf, wsize - 6);  /* 6 = END\r\n\0 */
        if (res == wsize - 6) {
            out_string(c, "SERVER_ERROR buffer overflow");
            free(wbuf); close(fd);
            return;
        }
        if (res == 0 || res == -1) {
            out_string(c, "SERVER_ERROR can't read the maps file");
            free(wbuf); close(fd);
            return;
        }
        strcpy(wbuf + res, "END\r\n");
        c->write_and_free=wbuf;
        c->wcurr=wbuf;
        c->wbytes = res + 6;
        c->state = conn_write;
        c->write_and_go = conn_read;
        close(fd);
        return;
    }

#if 0
    if (strcmp(command, "stats items")==0) {
        char buffer[4096];
        item_stats(buffer, 4096);
        out_string(c, buffer);
        return;
    }
#endif

#if 0
    if (strcmp(command, "stats sizes")==0) {
        int bytes = 0;
        char *buf = item_stats_sizes(&bytes);
        if (! buf) {
            out_string(c, "SERVER_ERROR out of memory");
            return;
        }

        c->write_and_free = buf;
        c->wcurr = buf;
        c->wbytes = bytes;
        c->state = conn_write;
        c->write_and_go = conn_read;
        return;
    }
#endif

    out_string(c, "ERROR");
}


/**
 * Main command loop, check the command passed on the received line
 */
void process_command ( conn *c, char *command ) 
{   
    int		comm;
    int		incr;
	int		i;
    int		res;
    int		locked;
    int		qlen;
    int		len;
    int		datasz;
	char *	pars;
	char	key[251];
	char	qkey[280];
    t_item *it;
    t_item *qit;
    time_t	exptime;
 
	comm = 0;
	incr = 0;
	
    /* 
     * for commands set/add/replace, we build an item and read the data
     * directly into it, then continue in nread_complete().
     */ 

    if (settings.verbose > 1)
	{
        fprintf(stderr, "<%d %s\n", c->sfd, command);
	}
	
    /* All incoming commands will require a response, so we cork at the beginning,
     * and uncork at the very end (usually by means of out_string)  
	 */
    set_cork(c, 1);

	/* Try to recognise the command, translate it to a comm command */
	i    = 0;
	pars = "";
	while (cmds[i].cmd_s != NULL)
	{
		if (strncmp(cmds[i].cmd_s, command, strlen(cmds[i].cmd_s)) == 0)
		{
			pars = command + strlen(cmds[i].cmd_s);
			while (*pars == ' ')
			{
				pars++;
			}
			break;
		}
		i++;
	}
	comm = cmds[i].comm;
	
	switch (comm)
	{
	case C_SET:
		key[0] = 0;
        res    = sscanf(pars, "%250s %u %ld %d%n", key, &locked, &exptime, &datasz, &len);
        if (res != 4 || strlen(key)==0) 
		{
            out_string(c, "CLIENT_ERROR bad command line format");
            return;
        }
        exptime = realtime(exptime);
        it      = item_alloc(key, pars+len, locked, exptime, datasz+2);
        if (!it) 
		{
            out_string(c, "SERVER_ERROR out of memory");

            /* swallow the data line */
            c->write_and_go = conn_swallow;
            c->sbytes       = len+2;
        }
        else
        {
	        c->item_comm  = comm;
	        c->item       = it;
	        c->rcurr      = it->data;
	        c->rlbytes    = it->datasz;
	        c->state      = conn_nread;
	        c->rqueue_len = 0;
	        strcpy(c->rqueue_key, "");
        }
        break;
		
	case C_QUEUE:
		key[0] = 0;
        res    = sscanf(pars, "%250s %u %u %ld %d%n", key, &qlen, &locked, &exptime, &datasz, &len);
        if (res != 5 || strlen(key)==0) 
		{
            out_string(c, "CLIENT_ERROR bad command line format");
            return;
        }
        
        if (qlen <= 0)
        {
            out_string(c, "CLIENT_ERROR bad command line format");
	
            /* swallow the data line */
            c->write_and_go = conn_swallow;
            c->sbytes       = len+2;
        }
        else
        {
			sprintf(qkey, "%s##%d", key, queue_serial_nr++);
	        exptime = realtime(exptime);
	        it      = item_alloc(qkey, pars+len, locked, exptime, datasz+2);
	        if (!it) 
			{
	            out_string(c, "SERVER_ERROR out of memory");
	
	            /* swallow the data line */
	            c->write_and_go = conn_swallow;
	            c->sbytes       = len+2;
	        }
	        else
	        {

				strcpy(c->rqueue_key, key);
					
		        c->item_comm  = comm;
		        c->item       = it;
		        c->rcurr      = it->data;
		        c->rlbytes    = it->datasz;
		        c->state      = conn_nread;
		        c->rqueue_len = qlen;
		        strcpy(c->rqueue_key, key);
			}
        }
		break;
		
	case C_DEQUEUE:
		/* fetch the list of keys to be fetched */
		i = 0;
        while (sscanf(pars, "%250s%n", key, &len) >= 1) 
		{
			pars += len;
            stats.get_cmds++;

            it = item_find(key);
            if (it && it->queue_len > 0) 
			{
				int j;
				
				/* Send the oldest item back, dequeue it */
				for (j=it->queue_len-1; j>=0; j--)
				{
					if (it->queue_items[j])
					{
		                if (i >= c->isize) 
						{
		                    t_item **new_list = (t_item **) realloc(c->ilist, sizeof(t_item *)*c->isize*2);
		                    if (new_list) 
							{
		                        c->isize *= 2;
		                        c->ilist  = new_list;
		                    }
							else
							{
								break;
							}
						}
		                c->ilist[i++]      = it->queue_items[j];
						it->queue_items[j] = NULL;
		                break;
					}
				}
                stats.get_hits++;
            }
			else
			{
				stats.get_misses++;
			}
		}
		
		/* set the counters of all items to be sent */
        c->icurr = c->ilist;
        c->ileft = i;

        if (c->ileft)
		{
            c->ipart  = 0;
            c->state  = conn_mwrite;
            c->ibytes = 0;
        }
		else
		{
            out_string(c, "END");
        }
		break;
		
	case C_GET:
		/* fetch the list of keys to be fetched */
		i = 0;
        while (sscanf(pars, "%250s%n", key, &len) >= 1) 
		{
            pars += len;
            stats.get_cmds++;

            it = item_find(key);
            if (it) 
			{
				if (it->queue_len)
				{
					int j;

					// Append all queued items to the send queue
					for (j=it->queue_len-1; j>=0; j--)
					{
						if (it->queue_items[j])
						{
			                if (i >= c->isize) 
							{
			                    t_item **new_list = (t_item **) realloc(c->ilist, sizeof(t_item *)*c->isize*2);
			                    if (new_list) 
								{
			                        c->isize *= 2;
			                        c->ilist  = new_list;
			                    }
								else
								{
									break;
								}
							}
			                item_ref(it->queue_items[j]);
			                c->ilist[i++] = it->queue_items[j];
						}
					}
				}
				else
				{
					// Single item, add it to the send queue
	                if (i >= c->isize) 
					{
	                    t_item **new_list = (t_item **) realloc(c->ilist, sizeof(t_item *)*c->isize*2);
	                    if (new_list) 
						{
	                        c->isize *= 2;
	                        c->ilist  = new_list;
	                    }
						else
						{
							break;
						}
					}
	                item_ref(it);
	                c->ilist[i++] = it;
				}
                stats.get_hits++;
            }
			else
			{
				stats.get_misses++;
			}
		}
		
		/* set the counters of all items to be sent */
        c->icurr = c->ilist;
        c->ileft = i;

        if (c->ileft)
		{
            c->ipart  = 0;
            c->state  = conn_mwrite;
            c->ibytes = 0;
        }
		else
		{
            out_string(c, "END");
        }
		break;
		
	case C_DELETE:
		exptime = 0;
		key[0]	= 0;
        res		= sscanf(pars, "%250s %ld", key, &exptime);

		if (strlen(key) > 0)
		{
			it = item_find(key);
			if (it)
			{
				if (exptime == 0) 
				{
					/* immediate delete 
					 */
					item_remove(it->h.key);
					out_string(c, "DELETED");
				}
				else
				{
					/* Delayed delete 
					 */
					if (delcurr >= deltotal) 
					{
						/* Try to expand the deletion queue */
						t_item **new_delete = realloc(todelete, sizeof(t_item *) * deltotal * 2);

						if (new_delete)
						{
							todelete  = new_delete;
							deltotal *= 2;
						}
					}

					if (delcurr < deltotal) 
					{
						exptime = realtime(exptime);
	
						/* Use its expiration time as its deletion time now 
						 */
						it->exptime			= exptime;
						todelete[delcurr++] = it;
						item_ref(it);
	
						out_string(c, "DELETED [queued]");
					}
					else
					{ 
						/* 
						 * User wants a delay but we ran out of memory for the delete queue.
						 * So we delete it immediately
						 */
						item_remove(it->h.key);
						out_string(c, "DELETED [immediate as out of memory]");
					}
				}
			}
			else
			{
				out_string(c, "NOT_FOUND");
			}
		}
		break;

	case C_STATS:
        process_stat(c, command);
		break;

	case C_FLUSH_ALL:
		dep_touch_all();
        out_string(c, "OK");
		break;

	case C_FLUSH:
		key[0]	= 0;
        res		= sscanf(pars, "%250s", key);
		if (strlen(key) > 0)
		{
			dep_touch(key);
			out_string(c, "OK");
		}
		else
		{
			out_string(c, "CLIENT_ERROR bad command line format");
		}
		break;
		
	case C_VERSION:
        out_string(c, "VERSION " VERSION);
		break;

	case C_QUIT:
        c->state = conn_closing;
		break;

	case C_ERROR:
	default:
		out_string(c, "ERROR unknown command");
		break;
	}
}

/**
 * If we have a complete line in the buffer, process it and move whatever
 * remains in the buffer to its beginning. 
 * Return 1 when a command is handled, otherwise 0.
 */
int try_read_command ( conn *c )
{
    char *	el;
    char *	cont;
	int		handled;
	
    if (c->rbytes > 0)
    {
	    el = memchr(c->rbuf, '\n', c->rbytes);
	    if (el)
	    {
		    cont = el + 1;
		    if (el - c->rbuf > 1 && *(el - 1) == '\r') 
		    {
		        el--;
		    }
		    *el = '\0';
		
		    process_command(c, c->rbuf);
		
		    if (cont - c->rbuf < c->rbytes)
		    {
		    	/* more stuff in the buffer */
		        memmove(c->rbuf, cont, c->rbytes - (cont - c->rbuf));
		    }
		    c->rbytes -= (cont - c->rbuf);
		    handled    = 1;
	    }
	    else
	    {
	    	handled = 0;
	    }
    }
    else
    {
    	handled = 0;
    }
    return handled;
}


/**
 * read from network as much as we can, handle buffer overflow and connection
 * close. 
 * return 0 if there's nothing to read on the first read.
 */
int try_read_network ( conn *c ) 
{
    int gotdata;
    int res;
	int reading;
	
	reading = 1;
	gotdata = 0;
 
    while (reading) 
    {
        if (c->rbytes >= c->rsize) 
        {
            char *new_rbuf;
            
            new_rbuf = realloc(c->rbuf, c->rsize*2);
            if (!new_rbuf)
            {
                if (settings.verbose > 0)
                {
                    fprintf(stderr, "Couldn't realloc input buffer\n");
                }
                c->rbytes       = 0; /* ignore what we read */
                out_string(c, "SERVER_ERROR out of memory");
                c->write_and_go = conn_closing;

                return 1;
            }
            c->rbuf   = new_rbuf;
            c->rsize *= 2;
        }

		/* Read the next bunch of data into the connection buffer */
        res = read(c->sfd, c->rbuf + c->rbytes, c->rsize - c->rbytes);

        if (res > 0)
        {
        	/* received some data */
            stats.bytes_read += res;
            c->rbytes        += res;
            gotdata           = 1;
        }
        else if (res == 0) 
	    {
	    	/* connection closed */
	        c->state = conn_closing;
	        gotdata  = 1;
	        reading  = 0;
	    }
		else if (res == -1) 
	    {
	    	/* error while reading data */
        	reading = 0;
	        if (errno != EAGAIN && errno != EWOULDBLOCK)
 	        {
				gotdata = 0;
	        }
        }
    }
    return gotdata;
}


int update_event ( conn *c, int new_flags ) 
{
    if (c->ev_flags == new_flags)
    {
        return 1;
    }
    if (event_del(&c->event) == -1) 
    {
    	return 0;
    }
    event_set(&c->event, c->sfd, new_flags, event_handler, (void *)c);
    c->ev_flags = new_flags;
    if (event_add(&c->event, 0) == -1)
    {
    	return 0;
    }
    return 1;
}


/**
 * The main state machine for a connection
 */
void drive_machine(conn *c) 
{
    int				exit = 0;
    int 			sfd;
    int				flags = 1;
    socklen_t		addrlen;
    struct sockaddr addr;
    conn *			newc;
    int 			res;

    while (!exit) 
    {
        /* printf("state %d\n", c->state);*/
        switch (c->state) 
        {
        case conn_listening:
            addrlen = sizeof(addr);
            if ((sfd = accept(c->sfd, &addr, &addrlen)) == -1) 
            {
                if (errno == EAGAIN || errno == EWOULDBLOCK) 
                {
                    exit = 1;
                }
                else
                {
                    perror("accept()");
                }
            }
			else
			{
				flags = fcntl(sfd, F_GETFL, 0);
	            if (	flags < 0
	            	||	fcntl(sfd, F_SETFL, flags | O_NONBLOCK) < 0) 
	            {
	                perror("setting O_NONBLOCK");
	                close(sfd);
	            }            
	            else
	            {
		            newc = conn_new(sfd, conn_read, EV_READ | EV_PERSIST);
		            if (!newc) 
		            {
		                if (settings.verbose > 0)
		                {
		                    fprintf(stderr, "couldn't create new connection\n");
		                }
		                close(sfd);
		            }
	            }
			}
            break;

        case conn_read:
            if (	!try_read_command(c) 
            	&&	!try_read_network(c)) 
            {
	            /* we have no command line and no data to read from network */
	            if (!update_event(c, EV_READ | EV_PERSIST)) 
	            {
	                if (settings.verbose > 0)
	                {
	                    fprintf(stderr, "Couldn't update event\n");
	                }
	                c->state = conn_closing;
	            }
	            else
	            {
		            exit = 1;
	            }
            }
            break;

        case conn_nread:
            /* we are reading rlbytes into rcurr; */
            if (c->rlbytes == 0) 
            {
                complete_nread(c);
            }
            else if (c->rbytes > 0) 
            {
	            /* first check if we have leftovers in the conn_read buffer */
                int tocopy  = (c->rbytes > c->rlbytes) ? c->rlbytes : c->rbytes;
                
                memcpy(c->rcurr, c->rbuf, tocopy);
                c->rcurr   += tocopy;
                c->rlbytes -= tocopy;
                if (c->rbytes > tocopy) 
                {
                    memmove(c->rbuf, c->rbuf+tocopy, c->rbytes - tocopy);
                }
                c->rbytes -= tocopy;
            }
			else
			{
	            /*  now try reading from the socket */
	            res = read(c->sfd, c->rcurr, c->rlbytes);
	            if (res > 0) {
	                stats.bytes_read += res;
	                c->rcurr += res;
	                c->rlbytes -= res;
	            }
	            else if (res == 0) 
	            { 
	            	/* end of stream */
	                c->state = conn_closing;
	            }
	            else if (res == -1 && (errno == EAGAIN || errno == EWOULDBLOCK)) 
	            {
	                if (!update_event(c, EV_READ | EV_PERSIST)) 
	                {
	                    if (settings.verbose > 0) 
	                    {
	                        fprintf(stderr, "Couldn't update event\n");
	                    }
	                    c->state = conn_closing;
	                }
	                else
	                {
		                exit = 1;
		            }
	            }
	            else
	            {
		            /* otherwise we have a real error, on which we close the connection */
		            if (settings.verbose > 0)
		            {
		                fprintf(stderr, "Failed to read, and not due to blocking\n");
		            }
		            c->state = conn_closing;
	            }
			}
            break;

        case conn_swallow:
            /* we are reading sbytes and throwing them away */
            if (c->sbytes == 0) 
            {
                c->state = conn_read;
            }
            else if (c->rbytes > 0) 
            {
	            /* first check if we have leftovers in the conn_read buffer */
                int tocopy = (c->rbytes > c->sbytes) ? c->sbytes : c->rbytes;
                
                c->sbytes -= tocopy;
                if (c->rbytes > tocopy) 
                {
                    memmove(c->rbuf, c->rbuf+tocopy, c->rbytes - tocopy);
                }
                c->rbytes -= tocopy;
            }
			else
			{
	            /*  now try reading from the socket */
	            res = read(c->sfd, c->rbuf, c->rsize > c->sbytes ? c->sbytes : c->rsize);
	            
	            if (res > 0) 
	            {
	                stats.bytes_read += res;
	                c->sbytes -= res;
	            }
	            else if (res == 0) 
	            {
	            	/* end of stream */
	                c->state = conn_closing;
	            }
	            else if (res == -1 && (errno == EAGAIN || errno == EWOULDBLOCK)) 
	            {
	                if (!update_event(c, EV_READ | EV_PERSIST)) 
	                {
	                    if (settings.verbose > 0)
	                    {
	                        fprintf(stderr, "Couldn't update event\n");
	                    }
	                    c->state = conn_closing;
	                }
	                else
	                {
		                exit = 1;
	                }
	            }
	            else
	            {
		            /* otherwise we have a real error, on which we close the connection */
		            if (settings.verbose > 0)
		            {
		            	fprintf(stderr, "Failed to read, and not due to blocking\n");
		            }
		            c->state = conn_closing;
	            }
			}
            break;

        case conn_write:
            /* we are writing wbytes bytes starting from wcurr */
            if (c->wbytes == 0) 
            {
                if (c->write_and_free) 
                {
                    free(c->write_and_free);
                    c->write_and_free = 0;
                }
                c->state = c->write_and_go;
                if (c->state == conn_read)
                {
                	set_cork(c, 0);
                }
            }
            else
            {
	            res = write(c->sfd, c->wcurr, c->wbytes);
	            if (res > 0) 
	            {
	                stats.bytes_written += res;
	                c->wcurr  += res;
	                c->wbytes -= res;
	            }
	            else if (res == -1 && (errno == EAGAIN || errno == EWOULDBLOCK)) 
	            {
	                if (!update_event(c, EV_WRITE | EV_PERSIST))
	                {
	                    if (settings.verbose > 0)
	                    {
	                        fprintf(stderr, "Couldn't update event\n");
	                    }
	                    c->state = conn_closing;
	                }
	                else
	                {
	                	exit = 1;
	                }
	            }
	            else
	            {
		            /* if res==0 or res==-1 and error is not EAGAIN or EWOULDBLOCK,
		               we have a real error, on which we close the connection */
		            if (settings.verbose > 0)
		            {
		            	fprintf(stderr, "Failed to write, and not due to blocking\n");
		            }
		            c->state = conn_closing;
	            }
            }
            break;

        case conn_mwrite:
            /* 
             * we're writing ibytes bytes from iptr. iptr alternates between
             * ibuf, where we build a string "VALUE...", and ITEM_data(it) for the 
             * current item. When we finish a chunk, we choose the next one using 
             * ipart, which has the following semantics: 0 - start the loop, 1 - 
             * we finished ibuf, go to current ITEM_data(it); 2 - we finished ITEM_data(it),
             * move to the next item and build its ibuf; 3 - we finished all items, 
             * write "END".
             */
            if (c->ibytes > 0) 
            {
                res = write(c->sfd, c->iptr, c->ibytes);
                if (res > 0) 
                {
                    stats.bytes_written += res;
                    c->iptr				+= res;
                    c->ibytes			-= res;
                }
                else if (res == -1 && (errno == EAGAIN || errno == EWOULDBLOCK)) 
                {
                    if (!update_event(c, EV_WRITE | EV_PERSIST)) 
                    {
                        if (settings.verbose > 0)
                        {
                        	fprintf(stderr, "Couldn't update event\n");
                        }
                        c->state = conn_closing;
                    }
                    else
                    {
	                    exit = 1;
                    }
                }
                else
                {
	                /* if res==0 or res==-1 and error is not EAGAIN or EWOULDBLOCK,
	                   we have a real error, on which we close the connection */
	                if (settings.verbose > 0)
	                {
	                    fprintf(stderr, "Failed to write, and not due to blocking\n");
	                }
	                c->state = conn_closing;
                }
            }
            else
            {
                t_item *it;
                
                /* we finished a chunk, decide what to do next */
                switch (c->ipart) 
                {
                case 1:
					/* Send the data of a key/value pair */
                    it 			= *(c->icurr);
                    c->iptr		= it->data;
                    c->ibytes 	= it->datasz;
                    c->ipart 	= 2;
                    break;

                case 2:
                	/* Finished sending the key/value pair, move on to next */
                    it 			= *(c->icurr);
                    item_unref(it);				// unref for the write
                    item_unref(it);				// unref for the dequeue
                    c->ileft--;
                    if (c->ileft <= 0)
                    {
	                    if (settings.verbose > 1)
	                    {
	                        fprintf(stderr, ">%d finished sending keys\n", c->sfd);
	                    }
                        c->ipart = 3;
                        break;
                    }
                    else 
                    {
                        c->icurr++;
                    }
                    /* FALL THROUGH */
                case 0:
                	/* Tell the client a key/value pair is coming */
                    it			= *(c->icurr);
                    c->ibytes 	= sprintf(c->ibuf, "VALUE %s %u %u\r\n", it->h.key, it->locked, it->datasz - 2);
                    c->iptr		= c->ibuf;
                    c->ipart	= 1;
                    item_ref(it);
                    
                    if (settings.verbose > 1)
                    {
                        fprintf(stderr, ">%d sending key %s\n", c->sfd, it->h.key);
                    }
                    break;

                case 3:
                    out_string(c, "END");
                    break;
                }
            }
            break;

        case conn_closing:
            conn_close(c);
            exit = 1;
            break;
        }

    } /* while !exit */
}

/**
 * Main event handler for a connection, called by libevent event_loop()
 */
void event_handler ( int fd, short which, void *arg ) 
{
    conn *c;
    
	now		 = time(0);
    c		 = (conn *) arg;
    c->which = which;

    /* sanity */
    if (fd != c->sfd) 
    {
        if (settings.verbose > 0)
        {
            fprintf(stderr, "Catastrophic: event fd doesn't match conn fd!\n");
        }
        conn_close(c);
    }
    else
    {
	    /* do as much I/O as possible until we block */
	    drive_machine(c);
    }
    /* wait for next event */
}


/**
 * Get a new socket
 */
static int new_socket ( void ) 
{
    int sfd;
    int flags;

	sfd = socket(AF_INET, SOCK_STREAM, 0);
    if (sfd == -1) 
    {
        perror("socket()");
    }
    else
    {
		flags = fcntl(sfd, F_GETFL, 0);
	    if (	flags < 0 
	    	||	fcntl(sfd, F_SETFL, flags | O_NONBLOCK) < 0) 
	    {
	        perror("setting O_NONBLOCK");
	        close(sfd);
	        sfd = -1;
	    }
    }
    return sfd;
}


/**
 * Open the server socket and start listening.
 */
static int server_socket( int port ) 
{
    int 				sfd;
    struct linger 		ling = {0, 0};
    struct sockaddr_in	addr;
    int 				flags;
    
    flags 	= 1;
    sfd 	= new_socket();

    if (sfd != -1) 
    {
	    setsockopt(sfd, SOL_SOCKET, SO_REUSEADDR, &flags, sizeof(flags));
	    setsockopt(sfd, SOL_SOCKET, SO_KEEPALIVE, &flags, sizeof(flags));
	    setsockopt(sfd, SOL_SOCKET, SO_LINGER,    &ling,  sizeof(ling));
#if !defined(TCP_NOPUSH)
	    setsockopt(sfd, IPPROTO_TCP, TCP_NODELAY, &flags, sizeof(flags));
#endif
	
	    /* 
	     * the memset call clears nonstandard fields in some impementations
	     * that otherwise mess things up.
	     */
	    memset(&addr, 0, sizeof(addr));
	
	    addr.sin_family = AF_INET;
	    addr.sin_port   = htons(port);
	    addr.sin_addr   = settings.interface;
	    
	    if (bind(sfd, (struct sockaddr *) &addr, sizeof(addr)) == -1) 
	    {
	        perror("bind()");
	        close(sfd);
	        sfd = -1;
	    }
	    else if (listen(sfd, 1024) == -1) 
	    {
	        perror("listen()");
	        close(sfd);
	        sfd = -1;
	    }
    }
    return sfd;
}

/* invoke right before gdb is called, on assert */
void pre_gdb ( void ) 
{
    int i = 0;

    if (l_socket)
    {
    	close(l_socket);
    }
    for (i=3; i<=500; i++)
    {
    	close(i); /* so lame */
    }
    kill(getpid(), SIGABRT);
}


/**
 * Check the delayed deletion queue if there are items to be deleted.
 * Delete all items that have an expiry (delete) date in the past.
 * This function is restarted every 5 seconds.
 */
void delete_handler ( int fd, short which, void *arg ) 
{
    static int 			initialized = 0;
	static struct event deleteevent;

    struct timeval	t;
    int 			i; 
    int 			j;

	now = time(0);

	/* Set the timer for the next invocation of the delete_handler() */
    if (initialized) 
    {
        /* some versions of libevent don't like deleting events that don't exist,
           so only delete once we know this event has been added. */
        evtimer_del(&deleteevent);

		/* check if there are any items that should be deleted by now */	
	    for (i=0, j=0; i<delcurr; i++) 
	    {
	        if (todelete[i]->exptime < now) 
	        {
	            item_unref(todelete[i]);
	        }
	        else
	        {
				// Shift item in the queue, removing the previous item
	            todelete[j++] = todelete[i];
	        }
	    }
	    delcurr = j;
    }
    else
    {
        initialized = 1;
    }

    evtimer_set(&deleteevent, delete_handler, 0);
    t.tv_sec   = 5; 
    t.tv_usec  = 0;
    evtimer_add(&deleteevent, &t);
}



/**
 * Garbage collector, cleans up out of date items, selects items to be deleted
 * when the maximum memory usage has been reached.
 */
void gc_handler ( int fd, short which, void *arg ) 
{
    static int 			initialized = 0;
	static struct event gcevent;
    struct timeval	t;

	now = time(0);

	/* Set the timer for the next invocation of the delete_handler() */
    if (initialized) 
    {
        /* some versions of libevent don't like deleting events that don't exist,
           so only delete once we know this event has been added. */
        evtimer_del(&gcevent);
		item_gc(settings.maxbytes);
    }
    else
    {
        initialized = 1;
    }

    evtimer_set(&gcevent, gc_handler, 0);
    t.tv_sec   = 2; 
    t.tv_usec  = 0;
    evtimer_add(&gcevent, &t);
}


        
void usage ( void ) 
{
    printf(PACKAGE " " VERSION "\n");
    printf("-p <num>      port number to listen on\n");
    printf("-l <ip_addr>  interface to listen on, default is INDRR_ANY\n");
    printf("-d            run as a daemon\n");
    printf("-r            maximize core file limit\n");
    printf("-u <username> assume identity of <username> (only when run as root)\n");
    printf("-m <num>      max memory to use for items in megabytes, default is 64 MB\n");
    printf("-M            return error on memory exhausted (rather than removing items)\n");
    printf("-c <num>      max simultaneous connections, default is 1024\n");
    printf("-k            lock down all paged memory\n");
    printf("-v            verbose (print errors/warnings while in event loop)\n");
    printf("-vv           very verbose (also print client commands/reponses)\n");
    printf("-h            print this help and exit\n");
    printf("-i            print depcached and libevent license\n");
    printf("-P <file>     save PID in <file>, only used with -d option\n");
}


void usage_license(void) 
{
    printf(PACKAGE " " VERSION "\n\n");
    printf(
	"Copyright (c) 2007, Mediamatic Lab, Inc. <http://www.mediamatic.nl/>\n"
	"All rights reserved.\n"
	"\n"
	"Redistribution and use in source and binary forms, with or without\n"
	"modification, are permitted provided that the following conditions are\n"
	"met:\n"
	"\n"
	"    * Redistributions of source code must retain the above copyright\n"
	"notice, this list of conditions and the following disclaimer.\n"
	"\n"
	"    * Redistributions in binary form must reproduce the above\n"
	"copyright notice, this list of conditions and the following disclaimer\n"
	"in the documentation and/or other materials provided with the\n"
	"distribution.\n"
	"\n"
	"    * Neither the name of Mediamatic nor the names of its\n"
	"contributors may be used to endorse or promote products derived from\n"
	"this software without specific prior written permission.\n"
	"\n"
	"THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS\n"
	"\"AS IS\" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT\n"
	"LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR\n"
	"A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT\n"
	"OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,\n"
	"SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT\n"
	"LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,\n"
	"DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY\n"
	"THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT\n"
	"(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE\n"
	"OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.\n"
	"\n"
	"\n"
	"This product includes software developed by Danga Interactive and Niels Provos.\n"
	"\n"
	"[ memcached ]\n"
	"\n"
	"Copyright (c) 2003, Danga Interactive, Inc. <http://www.danga.com/>\n"
	"All rights reserved.\n"
	"\n"
	"Redistribution and use in source and binary forms, with or without\n"
	"modification, are permitted provided that the following conditions are\n"
	"met:\n"
	"\n"
	"    * Redistributions of source code must retain the above copyright\n"
	"notice, this list of conditions and the following disclaimer.\n"
	"\n"
	"    * Redistributions in binary form must reproduce the above\n"
	"copyright notice, this list of conditions and the following disclaimer\n"
	"in the documentation and/or other materials provided with the\n"
	"distribution.\n"
	"\n"
	"    * Neither the name of the Danga Interactive nor the names of its\n"
	"contributors may be used to endorse or promote products derived from\n"
	"this software without specific prior written permission.\n"
	"\n"
	"THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS\n"
	"\"AS IS\" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT\n"
	"LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR\n"
	"A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT\n"
	"OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,\n"
	"SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT\n"
	"LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,\n"
	"DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY\n"
	"THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT\n"
	"(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE\n"
	"OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.\n"
	"\n"
	"\n"
	"[ libevent ]\n"
	"\n"
	"Copyright 2000-2003 Niels Provos <provos@citi.umich.edu>\n"
	"All rights reserved.\n"
	"\n"
	"Redistribution and use in source and binary forms, with or without\n"
	"modification, are permitted provided that the following conditions\n"
	"are met:\n"
	"1. Redistributions of source code must retain the above copyright\n"
	"   notice, this list of conditions and the following disclaimer.\n"
	"2. Redistributions in binary form must reproduce the above copyright\n"
	"   notice, this list of conditions and the following disclaimer in the\n"
	"   documentation and/or other materials provided with the distribution.\n"
	"3. All advertising materials mentioning features or use of this software\n"
	"   must display the following acknowledgement:\n"
	"      This product includes software developed by Niels Provos.\n"
	"4. The name of the author may not be used to endorse or promote products\n"
	"   derived from this software without specific prior written permission.\n"
	"\n"
	"THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR\n"
	"IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES\n"
	"OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.\n"
	"IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,\n"
	"INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT\n"
	"NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,\n"
	"DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY\n"
	"THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT\n"
	"(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF\n"
	"THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.\n"
    );
}


void save_pid ( pid_t pid, char *pid_file ) 
{
    FILE *fp;

    if (pid_file)
	{
	    fp = fopen(pid_file,"w");
	    if (!fp) 
	    {
	        fprintf(stderr,"Could not open the pid file %s for writing\n",pid_file);
	    }
	    else
	    {
		    fprintf(fp,"%ld\n",(long) pid);
		    if (fclose(fp) == -1) 
		    {
		        fprintf(stderr,"Could not close the pid file %s.\n",pid_file);
		    }
	    }
	}
}

void remove_pidfile ( char *pid_file ) 
{
	if (pid_file && unlink(pid_file)) 
	{
		fprintf(stderr, "Could not remove the pid file %s.\n", pid_file);
	}
}


int main ( int argc, char **argv ) 
{
    int 			c;
    conn *			l_conn;
    struct in_addr	addr;
    int 			lock_memory = 0;
    int				daemonize   = 0;
    int 			maxcore     = 0;
    char *			username    = 0;
    struct passwd *	pw;
    struct sigaction sa;
    struct rlimit 	rlim;
    char *			pid_file    = NULL;

    /* init settings */
    settings_init();

	now = time(0);

    /* process arguments */
    while ((c = getopt(argc, argv, "p:m:Mc:k?hirvdl:u:P:")) != -1) 
    {
        switch (c) 
        {
        case 'p':
            settings.port = atoi(optarg);
            break;
        case 'm':
            settings.maxbytes = atoi(optarg)*1024*1024;
            break;
        case 'M':
            settings.evict_to_free = 0;
            break;
        case 'c':
            settings.maxconns = atoi(optarg);
            break;
        case 'h':
        case '?':
            usage();
            exit(0);
        case 'i':
            usage_license();
            exit(0);
        case 'k':
            lock_memory = 1;
            break;
        case 'v':
            settings.verbose++;
            break;
        case 'l':
            if (!inet_aton(optarg, &addr))
            {
                fprintf(stderr, "Illegal address: %s\n", optarg);
                return 1;
            }
            else
            {
                settings.interface = addr;
            }
            break;
        case 'd':
            daemonize = 1;
            break;
        case 'r':
            maxcore = 1;
            break;
        case 'u':
            username = optarg;
            break;
        case 'P':
            pid_file = optarg;
            break;
        default:
            fprintf(stderr, "Illegal argument \"%c\"\n", c);
            return 1;
        }
    }

    if (maxcore) 
    {
        struct rlimit rlim_new;

        /* 
         * First try raising to infinity; if that fails, try bringing
         * the soft limit to the hard. 
         */
        if (getrlimit(RLIMIT_CORE, &rlim)==0) 
        {
            rlim_new.rlim_cur = rlim_new.rlim_max = RLIM_INFINITY;
            if (setrlimit(RLIMIT_CORE, &rlim_new)!=0) 
            {
                /* failed. try raising just to the old max */
                rlim_new.rlim_cur = rlim.rlim_max;
                rlim_new.rlim_max = rlim.rlim_max;

                (void) setrlimit(RLIMIT_CORE, &rlim_new);
            }
        }
        /* 
         * getrlimit again to see what we ended up with. Only fail if 
         * the soft limit ends up 0, because then no core files will be 
         * created at all.
         */
           
        if (getrlimit(RLIMIT_CORE, &rlim) != 0 || rlim.rlim_cur == 0)
        {
            fprintf(stderr, "failed to ensure corefile creation\n");
            exit(1);
        }
    }
                        
    /* 
     * If needed, increase rlimits to allow as many connections
     * as needed.
     */
    
    if (getrlimit(RLIMIT_NOFILE, &rlim) != 0) 
    {
        fprintf(stderr, "failed to getrlimit number of files\n");
        exit(1);
    }
    else
    {
        int  maxfiles = settings.maxconns;
        
        if (rlim.rlim_cur < maxfiles) 
        {
        	rlim.rlim_cur = maxfiles + 3;
        }
        if (rlim.rlim_max < rlim.rlim_cur)
        {
            rlim.rlim_max = rlim.rlim_cur;
        }
        if (setrlimit(RLIMIT_NOFILE, &rlim) != 0) 
        {
            fprintf(stderr, "failed to set rlimit for open files. Try running as root or requesting smaller maxconns value.\n");
            exit(1);
        }
    }

    /* 
     * initialization order: first create the listening socket
     * (may need root on low ports), then drop root if needed,
     * then daemonise if needed, then init libevent (in some cases
     * descriptors created by libevent wouldn't survive forking).
     */

    /* create the listening socket and bind it */
    l_socket = server_socket(settings.port);
    if (l_socket == -1) 
    {
        fprintf(stderr, "failed to listen\n");
        exit(1);
    }

    /* lose root privileges if we have them */
    if (getuid() == 0 || geteuid() == 0) 
    {
        if (username==0 || *username=='\0') 
        {
            fprintf(stderr, "can't run as root without the -u switch\n");
            return 1;
        }
        
        pw = getpwnam(username);
        if (pw == 0) 
        {
            fprintf(stderr, "can't find the user %s to switch to\n", username);
            return 1;
        }
        if (setgid(pw->pw_gid) < 0 || setuid(pw->pw_uid) < 0) 
        {
            fprintf(stderr, "failed to assume identity of user %s\n", username);
            return 1;
        }
    }

    /* daemonize if requested */
    /* if we want to ensure our ability to dump core, don't chdir to / */
    if (daemonize) 
    {
        int res;
 
        res = daemon(maxcore, settings.verbose);
        if (res == -1) 
        {
            fprintf(stderr, "failed to daemon() in order to daemonize\n");
            return 1;
        }
    }


    /* initialize other stuff */
    item_init();
    dep_init();
    event_init();
    stats_init();
    conn_init();

    /* lock paged memory if needed */
    if (lock_memory) 
    {
#ifdef HAVE_MLOCKALL
        mlockall(MCL_CURRENT | MCL_FUTURE);
#else
        fprintf(stderr, "warning: mlockall() not supported on this platform.  proceeding without locing memory.\n");
#endif
    }

    /*
     * ignore SIGPIPE signals; we can use errno==EPIPE if we
     * need that information
     */
    sa.sa_handler = SIG_IGN;
    sa.sa_flags = 0;
    if (	sigemptyset(&sa.sa_mask)   == -1
       ||	sigaction(SIGPIPE, &sa, 0) == -1) 
    {
        perror("failed to ignore SIGPIPE; sigaction");
        exit(1); 
    }

    /* create the initial listening connection */
    if (!(l_conn = conn_new(l_socket, conn_listening, EV_READ | EV_PERSIST))) 
    {
        fprintf(stderr, "failed to create listening connection");
        exit(1);
    }

    /* initialise deletion array and timer event */
    deltotal = 200; 
    delcurr  = 0;
    todelete = malloc(sizeof(t_item *)*deltotal);

	/* sets up the periodic events */
    delete_handler(0,0,0); 	
    gc_handler(0,0,0);
    
    /* save the PID in if we're a daemon */
    if (daemonize)
    {
        save_pid(getpid(),pid_file);
    }
    
    /* enter the loop */
    event_loop(0);

    /* remove the PID file if we're a daemon */
    if (daemonize)
    {
        remove_pidfile(pid_file);
    }
    return 0;
}

