/* -*- Mode: C; tab-width: 4; c-basic-offset: 4; indent-tabs-mode: nil -*- */
/* $Id: item.c 30245 2007-08-20 16:57:10Z marc $ */

#include "config.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <assert.h>

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#include "hashtable.h"
#include "item.h"
#include "depcached.h"

#define ITEM_HASHPOWER 20

/* The hashtable for all items */
static t_hashtable * items 			= NULL;
static unsigned long item_bytes		= 0;

/* The garbage collector, active when item_bytes reaches 80% of the max memory size */
static unsigned	  	 gc_index		= 0;
static unsigned long gc_batch  		= 10000;	/* Number of hash entries checked with each gc run */
static t_item ** 	 gc_pool		= NULL;
static int			 gc_pool_use	= 0;
static int			 gc_pool_size	= 5000;		/* Size of garbage collector pool */

/* static function prototypes */
static int _item_valid ( t_item * it );


/**
 * Initialise the item hash table
 */
void item_init ( void )
{
	items = ht_new(ITEM_HASHPOWER);
	if (!items)
	{
		fprintf(stderr, "out of memory, could not allocate the items hashtable");
		exit(1);
	}
}


/**
 * Find an item, return the item when the dependencies and the modification date are ok
 */
t_item * item_find ( char * key )
{
	t_item * it;
	
	it = (t_item *) ht_find(items, key);
	if (it)
	{
		if (_item_valid(it))
		{
			if (it && it->gc)
			{
				it->gc = 0;
			}
		}
		else
		{
			item_remove(key);
			it = NULL;
		}
	}
	return it;
}


/**
 * Allocate a new item, splitting the key path, the dependency keys and setting the flags
 * The item is not yet inserted into the hashtable, as the data is still being received.
 */
t_item *item_alloc ( char *key, char *deps, int locked, time_t exptime, int nbytes )
{
	t_item * it;
	int		 i;
	
	it = malloc(sizeof(t_item));
	if (it)
	{
		if (nbytes > 0)
		{
			it->data = malloc(nbytes);
		}
		else
		{
			it->data = NULL;
		}

		if (it->data || nbytes == 0)
		{
			it->h.key	  = malloc(strlen(key)+1);
			it->h.h_next  = NULL;
			
			if (it->h.key)
			{
				char **p;
				char   d[20][251];
				int	   n;
				int	   d_ct;
				
				strcpy(it->h.key, key);
				it->datasz	    = nbytes;
				it->refcount    = 0;
				it->locked	    = locked;
				it->gc		    = 0;
				it->queue_len   = 0;
				it->queue_items = NULL;
				it->exptime	    = exptime;
				it->depct	    = 0;
				it->deps	    = NULL;
				it->depserial   = dep_serial();

				/* Add the dependencies	from the path of the variable, skip the complete path (last dep key) */		
				p = dep_path_split(key);
				i = 0;
				while (p[i] != NULL && p[i+1] != NULL)
				{
					i++;
				}

				/* Determine the dependencies from the given deps list */
				d_ct = 0;
				n    = 0;
				while (d_ct < 20 && sscanf(deps, "%250s%n", d[d_ct], &n) >= 1)
				{
					deps += n; 
					d_ct++;
				}

				/* Find all dependendcies and store refs to them */
				it->depct = i + d_ct;
				it->deps  = calloc(it->depct, sizeof(t_dep *));
				i         = 0;

				while (p[i] != NULL && p[i+1] != NULL)
				{
					it->deps[i] = dep_find_ref(p[i]);
					i++;
				}

				while (d_ct-- > 0)
				{
					it->deps[i] = dep_find_ref(d[d_ct]);
					i++;
				}
			}
			else
			{
				if (it->data)
				{
					free(it->data);
				}
				free(it);
				it = NULL;
			}
		}
		else
		{
			free(it);
			it = NULL;
		}
	}
	return it;
}



/**
 * Removes the item from the hashtable, frees all resources associated with the item when the
 * refcount reaches 0. 
 */
int item_remove ( char * key )
{
	t_item *it;
	
	it = (t_item *) ht_find(items, key);
	
	// Remove from the hashtable
	if (it)
	{
		ht_remove(items, key);
		item_unref(it);
	}
	return 0;
}



/**
 * Removes the item from the hashtable, frees all resources associated with the item when the
 * refcount reaches 0.   Only removes the item when it is in the hashtable.
 */
int item_remove_p ( t_item * rmit )
{
	t_item *it;
	
	it = (t_item *) ht_find(items, rmit->h.key);
	
	// Remove from the hashtable
	if (it == rmit)
	{
		ht_remove(items, rmit->h.key);
		item_unref(it);
		return 0;
	}
	else
	{
		return -1;
	}
}



/**
 * Replace an existing key with the given (new) item.  The new item should not be part of the hashtable yet.
 */
int item_replace ( t_item * new_it )
{
	int	ret;
	
	if (item_remove(new_it->h.key) == 0)
	{
		// previous item is removed, or there was no previous item
		ht_insert(items, (t_hashitem *) new_it);
		item_ref(new_it);

		item_bytes += sizeof(t_item) + new_it->datasz + strlen(new_it->h.key) + 1;
		ret			= 0;
	}
	else
	{
		// previous item could not be replaced (for whatever reason)
		ret = -1;
	}
	return ret;
}


/**
 * Increments the reference count of the item
 */
void item_ref ( t_item *it )
{
	it->refcount++;
}


/**
 * Decrements the refcount of the item, deletes it iff the delete flag is set and the item refcount reaches zero
 */
void item_unref ( t_item *it )
{
	if (it->refcount)
	{
		it->refcount--;
	}
	
	if (it->refcount == 0)
	{
		int i;

		item_bytes -= sizeof(t_item) + it->datasz + strlen(it->h.key) + 1;
		
		// Unref all dependency keys - needed for the dependency garbage collector
		for (i=0; i<it->depct; i++)
		{
			dep_unref(it->deps[i]);
		}

		// Unref queued items (if the current item is a queue)
		for (i=0; i<it->queue_len; i++)
		{
			if (it->queue_items[i])
			{
				item_unref(it->queue_items[i]);
			}
		}
		
		// Free all allocated memory of this item
		free(it->h.key);
		if (it->deps)
		{
			free(it->deps);
		}
		if (it->queue_items)
		{
			free(it->queue_items);
		}
		if (it->data)
		{
			free(it->data);
		}
		free(it);
	}
}



/**
 * Garbage collection, removes out of date items and replenishes the gc pool with deletion candidates
 */
void item_gc ( unsigned long max_bytes )
{
	int i;
	int j;
	int n;
	int gc;
	int gc_pool_use1;

	if (gc_pool == NULL)
	{
		gc_pool = (t_item **) calloc(gc_pool_size, sizeof(t_item *));
	}

	/* Remove all items from the gc pool that have been used after they
	 * were placed in the gc pool
	 */	
	for (i=0, j=0; i<gc_pool_use; i++)
	{
		if (gc_pool[i])
		{
			if (!gc_pool[i]->gc || gc_pool[i]->refcount == 1)
			{
				/* used after placed in the gc pool, or only in the gc pool - unref it */
				item_unref(gc_pool[i]);
			}
			else
			{
				/* Keep in the gc pool */
				gc_pool[j] = gc_pool[i];
				j++;
			}
		}
	}
	gc_pool_use  = j;
	gc_pool_use1 = gc_pool_use;

	/* Above the threshold, start active garbage collection from the random gc pool.
	 * First populate the gc pool till it is filled (or no items are left).
	 */
	if (max_bytes/8 < item_bytes/10)
	{
		gc = 1;
	}
	else
	{
		gc =  0;
	}


	/* When examining candidates, delete all entries that are out of date or for
	 * which the dependency keys have been touched.
	 * 
	 * Every non-dropped item will be added to the gc pool, the hash function makes sure that this is
	 * a random process.  Locked items will be skipped.
	 */
	 
	/**
	 * step over the hashtable entries (starting at gc_index) to clear out old items 
	 * and (if needed) repopulate the gc pool
	 */
	
	for (n=gc_batch-1; n>=0; n--)
	{
		gc_index = (gc_index + 1) % items->size;
		
		if (items->table[gc_index])
		{
			t_item **p;
			
			p = (t_item **) &items->table[gc_index];

			while (*p)
			{
				if (!_item_valid(*p))
				{
					/* Item is not valid anymore (date or dependency check failed) */
					item_remove_p(*p);
				}
				else if (	gc 
						&&	gc_pool_use < gc_pool_size 
						&&	!(*p)->gc
						&&	!(*p)->locked)
				{
					/* Add to the gc pool */
					gc_pool[gc_pool_use++] = *p;
					item_ref(*p);

					p = (t_item **) &(*p)->h.h_next;
				}
				else
				{
					if ((*p)->queue_len > 0)
					{
						/* Check all queue items, when an item is not valid then we can dequeue it */
						j = 0;
						for (i=0; i<(*p)->queue_len; i++)
						{
							if (	(*p)->queue_items[i]
								&&	!_item_valid((*p)->queue_items[i]))
							{
								item_unref((*p)->queue_items[i]);
							}
							else
							{
								(*p)->queue_items[j] = (*p)->queue_items[i];
								j++;
							}
						}

						while (j<(*p)->queue_len)
						{
							(*p)->queue_items[j++] = NULL;
						}
					}
					p = (t_item **) &(*p)->h.h_next;
				}
			}
		}
	}
	
	/**
	 *  when the allocated bytes are still too high, start deleting from the gc pool
	 * the deletion may only be done from the first gc_pool_use candidates (to prevent
	 * deletion of heavily used items).
	 */
	if (max_bytes/8 < item_bytes/10)
	{
		j = 0;
		for (i=0; i<gc_pool_use1; i++)
		{
			/* 25% chance that gc pool entries are dropped */
			if (random() % 4 == 0)
			{
				item_unref(gc_pool[i]);			/* remove from gc_pool */
				item_remove_p(gc_pool[i]);		/* remove item from items */
			}
			else
			{
				gc_pool[j++] = gc_pool[i];
			}
		}
		
		while (i<gc_pool_use)
		{
			gc_pool[j++] = gc_pool[i++];
		}
		gc_pool_use = j;
	}
}



/**
 * Check if an item is valid
 */
static int _item_valid ( t_item * it )
{
	extern time_t now;
	int valid;
	
	if (it->exptime != 0 && it->exptime < now)
	{
		valid = 0;
	}
	else
	{
		int 		i;
		t_depser	my_ser;
		
		my_ser = it->depserial;
		
		if (my_ser <= dep_serial_low())
		{
			valid = 0;
		}
		else
		{
			valid  = 1;
			for (i=0; valid && it != NULL && i<it->depct; i++)
			{
				if (it->deps[i]->depserial > my_ser)
				{
					extern struct settings settings;

					if (settings.verbose > 0)
					{
					    fprintf(stderr, "-- item invalid because of %s\n", it->deps[i]->h.key);
					}
					valid = 0;
				}
			}
		}
	}
	return valid;
}


