/* -*- Mode: C; tab-width: 4; c-basic-offset: 4; indent-tabs-mode: nil -*- */
/*
 * Hash table
 *
 * The hash function used here is by Bob Jenkins, 1996:
 *    <http://burtleburtle.net/bob/hash/doobs.html>
 *       "By Bob Jenkins, 1996.  bob_jenkins@burtleburtle.net.  
 *       You may use this code any way you wish, private, educational, 
 *       or commercial.  It's free."
 *
 * The rest of the file is licensed under the BSD license.  See LICENSE.
 *
 * $Id: hashtable.c 27705 2007-04-27 14:25:39Z marc $
 */

#include "config.h"

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <assert.h>

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#include "hashtable.h"

/* hard-code one million buckets, for now (2**20 == 4MB hash) */
#define HASHPOWER  20

#define hashsize(n) ((ub4)1<<(n))
#define hashmask(n) (hashsize(n)-1)

/* some type definitions, for known length integers */
typedef  unsigned long  int  ub4;   /* unsigned 4-byte quantities */
typedef  unsigned       char ub1;   /* unsigned 1-byte quantities */


/* function prototypes */
static ub4				hash             ( register ub1 *k, register ub4  length, register ub4  initval);
static t_hashitem ** 	_hashitem_before ( t_hashtable *ht, char *key );


/**
 * Allocate the empty hashtable of HASHPOWER entries.
 */
t_hashtable * ht_new ( int power )
{
    unsigned int	hash_size;
	t_hashtable *	ht;
	
	ht = malloc(sizeof(t_hashtable));
	if (ht)
	{
		ht->power = power;
		ht->size  = hashsize(power);
		ht->mask  = hashmask(power);
		ht->load  = 0;
		ht->table = calloc(ht->size, sizeof(void*));
    }
	if (!ht || !ht->table)
	{
        fprintf(stderr, "Failed to init hashtable.\n");
        exit(1);
    }
	return ht;
}


/**
 * Find a hashtable entry using the given key.
 * Returns either NULL or the hashtable item with the correct key.
 */
t_hashitem * ht_find ( t_hashtable *ht, char *key )
{
    ub4			hv;
    t_hashitem *it;
	
	hv = hash((unsigned char *) key, strlen(key), 0) & ht->mask;
	it = ht->table[hv];
    
    while (it && strcmp(key, it->key) != 0) 
	{
		it = it->h_next;
    }
    return it;
}



/**
 * Insert an item into the hash table, no duplicate checks are performed. The key field of the item
 * must be set.
 * Returns -1 when the key cound not be inserted (prob out of memory)
 */
int ht_insert ( t_hashtable *ht, t_hashitem *it ) 
{
    ub4 hv;
	
	if (it->key)
	{
		hv 			  = hash((unsigned char *) it->key, strlen(it->key), 0) & ht->mask;
		it->h_next    = ht->table[hv];
		ht->table[hv] = it;
		ht->load++;
		return 0;
	}
	else
	{
		return -1;
	}
}


/**
 * Remove the item with the given key, deallocates the hashtable key
 */
t_hashitem * ht_remove ( t_hashtable *ht, char *key ) 
{
    t_hashitem ** before;
	t_hashitem *  removed;
	
	before = _hashitem_before(ht, key);
    if (*before) 
	{
        t_hashitem * next;

		removed			= *before;
		next			= removed->h_next;
        removed->h_next	= 0;
        *before			= next;
		ht->load--;
	}
	else
	{
		removed = NULL;
	}
	return removed;
}



/**************************** static functions ******************************/

/**
 * Returns the address of the item pointer before the key.  if *item == 0, the item wasn't found 
 */
static t_hashitem ** _hashitem_before ( t_hashtable *ht, char *key ) 
{
    ub4				hv;
    t_hashitem **	pos;

	hv  = hash((unsigned char *) key, strlen(key), 0) & ht->mask;
	pos = &ht->table[hv];
	
    while (*pos && strcmp(key, (*pos)->key))
	{
        pos = &(*pos)->h_next;
    }
    return pos;
}





#define mix(a,b,c) \
{ \
  a -= b; a -= c; a ^= (c>>13); \
  b -= c; b -= a; b ^= (a<<8); \
  c -= a; c -= b; c ^= (b>>13); \
  a -= b; a -= c; a ^= (c>>12);  \
  b -= c; b -= a; b ^= (a<<16); \
  c -= a; c -= b; c ^= (b>>5); \
  a -= b; a -= c; a ^= (c>>3);  \
  b -= c; b -= a; b ^= (a<<10); \
  c -= a; c -= b; c ^= (b>>15); \
}

/*
--------------------------------------------------------------------
hash() -- hash a variable-length key into a 32-bit value
  k       : the key (the unaligned variable-length array of bytes)
  len     : the length of the key, counting by bytes
  initval : can be any 4-byte value
Returns a 32-bit value.  Every bit of the key affects every bit of
the return value.  Every 1-bit and 2-bit delta achieves avalanche.
About 6*len+35 instructions.

The best hash table sizes are powers of 2.  There is no need to do
mod a prime (mod is sooo slow!).  If you need less than 32 bits,
use a bitmask.  For example, if you need only 10 bits, do
  h = (h & hashmask(10));
In which case, the hash table should have hashsize(10) elements.

If you are hashing n strings (ub1 **)k, do it like this:
  for (i=0, h=0; i<n; ++i) h = hash( k[i], len[i], h);

By Bob Jenkins, 1996.  bob_jenkins@burtleburtle.net.  You may use this
code any way you wish, private, educational, or commercial.  It's free.

See http://burtleburtle.net/bob/hash/evahash.html
Use for hash table lookup, or anything where one collision in 2^^32 is
acceptable.  Do NOT use for cryptographic purposes.
--------------------------------------------------------------------
*/

static ub4 hash( k, length, initval)
     register ub1 *k;        /* the key */
     register ub4  length;   /* the length of the key */
     register ub4  initval;  /* the previous hash, or an arbitrary value */
{
    register ub4 a,b,c,len;

    /* Set up the internal state */
    len = length;
    a = b = 0x9e3779b9;  /* the golden ratio; an arbitrary value */
    c = initval;         /* the previous hash value */

    /*---------------------------------------- handle most of the key */
    while (len >= 12)
        {
            a += (k[0] +((ub4)k[1]<<8) +((ub4)k[2]<<16) +((ub4)k[3]<<24));
            b += (k[4] +((ub4)k[5]<<8) +((ub4)k[6]<<16) +((ub4)k[7]<<24));
            c += (k[8] +((ub4)k[9]<<8) +((ub4)k[10]<<16)+((ub4)k[11]<<24));
            mix(a,b,c);
            k += 12; len -= 12;
        }

    /*------------------------------------- handle the last 11 bytes */
    c += length;
    switch(len)              /* all the case statements fall through */
        {
        case 11: c+=((ub4)k[10]<<24);
        case 10: c+=((ub4)k[9]<<16);
        case 9 : c+=((ub4)k[8]<<8);
            /* the first byte of c is reserved for the length */
        case 8 : b+=((ub4)k[7]<<24);
        case 7 : b+=((ub4)k[6]<<16);
        case 6 : b+=((ub4)k[5]<<8);
        case 5 : b+=k[4];
        case 4 : a+=((ub4)k[3]<<24);
        case 3 : a+=((ub4)k[2]<<16);
        case 2 : a+=((ub4)k[1]<<8);
        case 1 : a+=k[0];
            /* case 0: nothing left to add */
        }
    mix(a,b,c);
    /*-------------------------------------------- report the result */
    return c;
}


