/* -*- Mode: C; tab-width: 4; c-basic-offset: 4; indent-tabs-mode: nil -*- */
/* $Id: dep.c 27925 2007-05-15 15:39:15Z marc $ */

#include "config.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#include "hashtable.h"
#include "dep.h"
#include "depcached.h"

#define DEP_HASHPOWER 18

static t_hashtable *	deps			= NULL;
static t_depser			depserial		= 1;
static t_depser			depserial_low	= 0;


/**
 * Initialize the dependency hashtables
 */
void dep_init ( void )
{
	deps = ht_new(DEP_HASHPOWER);
	if (!deps)
	{
		fprintf(stderr, "out of memory when allocating the dependency hash table");
		exit(1);
	}
}


/**
 * Touch the given dependency key, creates the key when needed
 */
int	dep_touch ( char *key )
{
	t_dep * d;
	int     k;
	extern struct settings settings;

	d = dep_find(key);
	if (d)
	{
		if (settings.verbose > 0)
		{
		    fprintf(stderr, "-- flushing key %s\n", key);
		}
		d->depserial = ++depserial;
	}

	/* Just to be sure: also flush the key without trailing '/' */
	k = strlen(key) - 1;
	if (k > 0 && key[k] == '/')
	{
		while (k >= 0 && key[k] == '/')
		{
			key[k] = '\0';
			k--;
		}
		if (k > 0)
		{
			d = dep_find(key);
			if (d)
			{
				if (settings.verbose > 0)
				{
				    fprintf(stderr, "-- flushing key %s\n", key);
				}
				d->depserial = ++depserial;
			}
		}
	}
}


/**
 * Sets the dependency low water mark, every item before this low water mark should be
 * considered deleted (or out of date).
 */
void dep_touch_all ( void )
{
	depserial_low = depserial;
	depserial++;
}


/**
 * Find the given dependency key, creates the key when needed
 */
t_dep * dep_find ( char *key )
{
	t_dep * d;
	
	d = (t_dep *) ht_find(deps, key);
	if (!d)
	{
		d = malloc(sizeof(t_dep));
		if (d)
		{
			d->h.key	 = malloc(strlen(key)+1);
			d->refcount  = 0;
			d->depserial = depserial;
			
			if (d->h.key)
			{
				strcpy(d->h.key, key);
				if (ht_insert(deps, (t_hashitem *) d) != 0)
				{
					free(d->h.key);
					free(d);
					d = NULL;
				}
			}
			else
			{
				free(d);
				d = NULL;
			}
		}
	}
	return d;
}


/**
 * Return the dependency entry for the key, increment reference count
 */
t_dep * dep_find_ref ( char * key )
{
	t_dep *d;
	
	d = dep_find(key);
	if (d)
	{
		d->refcount++;
	}
	return d;
}


/**
 * Decrement the refcount of the entry
 */
void dep_unref ( t_dep * d )
{
	if (d && d->refcount)
	{
		d->refcount--;
	}
}


/**
 * Return the current dependency serial nr
 */
t_depser dep_serial ( void )
{
	return depserial;
}


/**
 * Return the current dependency serial nr low water mark, everything before this low water mark
 * should be considered deleted.
 */
t_depser dep_serial_low ( void )
{
	return depserial_low;
}


/**
 * Split a path on '/' into parts.  The returned ref is a ref to an array, the array
 * is static so will be overwritten with the next invocation of this function.
 * The last part in the returned array is NULL
 */
char ** dep_path_split ( char * key )
{
	static char s[4002];
	static char *ps[32];
	
	int n;
	int i;
	int k;
	int offs;
	
	i    = 0;
	n    = 0;
	offs = 0;

	while (key[i] && offs+i < 4000 && n < 30)
	{
		if (key[i] == '/'&& i >= 1)
		{
			strncpy(s+offs, key, i);
			s[offs+i] = '\0';
			k         = i-1;
			while (k > 0 && s[offs+k] == '/')
			{
				s[offs+k] = '\0';
				k--;
			}

			if (	k >= 0
				&&	(n == 0 || strcmp(ps[n-1], s+offs)))
			{
				ps[n] = s+offs;
				offs += strlen(ps[n]) + 1;
				n++;
			}
		}
		i++;
	}

	// Take also the key itself
	if (	n < 30 
		&&	strlen(key) + offs + 1 < 4000
		&&	(n == 0 || strcmp(ps[n-1], key)))
	{
		strcpy(s+offs, key);
		ps[n++] = s+offs;
	}
	ps[n] = NULL;
	return ps;		
}

